@extends('layouts.admin-simple')

@section('title', 'Pending Approval - Outgoing Transactions')

@section('content')
<div class="container-fluid px-4">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1 class="h3 mb-2 text-gray-800">
                        <i class="material-icons align-middle mr-2">pending_actions</i>
                        Approval Queue
                    </h1>
                    <p class="text-muted mb-0">Review and approve pending outgoing transactions</p>
                </div>
                <div class="btn-group">
                    <a href="{{ route('warehouse.outgoing.index') }}" class="btn btn-secondary">
                        <i class="material-icons mr-1">arrow_back</i>
                        Back to Dashboard
                    </a>
                    <a href="{{ route('warehouse.outgoing.create') }}" class="btn btn-primary">
                        <i class="material-icons mr-1">add</i>
                        New Transaction
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Statistics Card -->
    <div class="row mb-4">
        <div class="col-lg-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Pending Approval"
                :value="$transactions->total()"
                icon="pending_actions"
                color="warning"
                subtitle="Awaiting manager approval"
            />
        </div>
        <div class="col-lg-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Today's Requests"
                :value="$transactions->where('transaction_date', today())->count()"
                icon="today"
                color="info"
                subtitle="Created today"
            />
        </div>
        <div class="col-lg-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="This Week"
                :value="$transactions->where('transaction_date', '>=', now()->startOfWeek())->count()"
                icon="date_range"
                color="primary"
                subtitle="Created this week"
            />
        </div>
        <div class="col-lg-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Urgent"
                :value="$transactions->where('transaction_date', '<=', now()->addDays(1))->count()"
                icon="priority_high"
                color="danger"
                subtitle="Need immediate attention"
            />
        </div>
    </div>

    <!-- Pending Transactions Table -->
    <div class="card">
        <div class="card-header">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="material-icons align-middle mr-1">view_list</i>
                Pending Approval Transactions ({{ $transactions->total() }})
            </h6>
        </div>
        <div class="card-body">
            @if($transactions->count() > 0)
                <div class="table-responsive">
                    <table class="table table-bordered table-hover">
                        <thead class="thead-light">
                            <tr>
                                <th>Reference</th>
                                <th>Date</th>
                                <th>Project</th>
                                <th>Division</th>
                                <th>Requested By</th>
                                <th>Type of Goods</th>
                                <th>Receiver</th>
                                <th>Days Pending</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($transactions as $transaction)
                                <tr class="{{ $transaction->transaction_date->lt(now()->subDays(2)) ? 'table-warning' : '' }}">
                                    <td>
                                        <strong>{{ $transaction->reference_number }}</strong>
                                        @if($transaction->material_request_number)
                                            <br><small class="text-muted">{{ $transaction->material_request_number }}</small>
                                        @endif
                                    </td>
                                    <td>
                                        {{ $transaction->transaction_date->format('M d, Y') }}
                                        <br><small class="text-muted">{{ $transaction->transaction_date->format('l') }}</small>
                                    </td>
                                    <td>
                                        @if($transaction->project)
                                            <strong>{{ $transaction->project->project_name }}</strong>
                                            @if($transaction->project->project_number)
                                                <br><small class="text-muted">{{ $transaction->project->project_number }}</small>
                                            @endif
                                        @else
                                            <span class="text-muted">N/A</span>
                                        @endif
                                    </td>
                                    <td>
                                        @if($transaction->division)
                                            {{ $transaction->division->division_name }}
                                        @else
                                            <span class="text-muted">N/A</span>
                                        @endif
                                    </td>
                                    <td>
                                        @if($transaction->requester)
                                            <strong>{{ $transaction->requester->name }}</strong>
                                            @if($transaction->requester->role)
                                                <br><small class="text-muted">{{ $transaction->requester->role->display_name ?? $transaction->requester->role->name }}</small>
                                            @endif
                                        @else
                                            <span class="text-muted">N/A</span>
                                        @endif
                                    </td>
                                    <td>{{ $transaction->type_of_goods }}</td>
                                    <td>{{ $transaction->receiver_name }}</td>
                                    <td>
                                        @php
                                            $daysPending = $transaction->created_at->diffInDays(now());
                                        @endphp
                                        <span class="badge {{ $daysPending > 2 ? 'badge-danger' : ($daysPending > 1 ? 'badge-warning' : 'badge-info') }}">
                                            {{ $daysPending }} {{ Str::plural('day', $daysPending) }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('warehouse.outgoing.show', $transaction) }}"
                                               class="btn btn-sm btn-outline-primary" title="View Details">
                                                <i class="material-icons" style="font-size: 16px;">visibility</i>
                                            </a>

                                            <!-- Approve Button -->
                                            <button type="button"
                                                    class="btn btn-sm btn-outline-success approve-btn"
                                                    title="Approve Transaction"
                                                    data-id="{{ $transaction->id }}"
                                                    data-reference="{{ $transaction->reference_number }}"
                                                    data-toggle="modal"
                                                    data-target="#approveModal">
                                                <i class="material-icons" style="font-size: 16px;">check</i>
                                            </button>

                                            <!-- Reject Button -->
                                            <button type="button"
                                                    class="btn btn-sm btn-outline-danger reject-btn"
                                                    title="Reject Transaction"
                                                    data-id="{{ $transaction->id }}"
                                                    data-reference="{{ $transaction->reference_number }}">
                                                <i class="material-icons" style="font-size: 16px;">close</i>
                                            </button>

                                            <a href="{{ route('warehouse.outgoing.edit', $transaction) }}"
                                               class="btn btn-sm btn-outline-secondary" title="Edit">
                                                <i class="material-icons" style="font-size: 16px;">edit</i>
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="d-flex justify-content-between align-items-center mt-4">
                    <div class="text-muted">
                        Showing {{ $transactions->firstItem() }} to {{ $transactions->lastItem() }} of {{ $transactions->total() }} results
                    </div>
                    {{ $transactions->links() }}
                </div>
            @else
                <div class="text-center py-5">
                    <i class="material-icons text-success mb-3" style="font-size: 48px;">check_circle</i>
                    <h5 class="text-muted">No Pending Approvals</h5>
                    <p class="text-muted">All outgoing transactions have been processed!</p>
                    <a href="{{ route('warehouse.outgoing.index') }}" class="btn btn-primary">
                        <i class="material-icons mr-1">dashboard</i>
                        View All Transactions
                    </a>
                </div>
            @endif
        </div>
    </div>

    <!-- Quick Actions Card -->
    @if($transactions->count() > 0)
    <div class="row mt-4">
        <div class="col-12">
            <div class="card border-left-warning">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                <i class="material-icons align-middle mr-1">speed</i>
                                Quick Actions
                            </div>
                            <div class="text-gray-800">
                                <button type="button" class="btn btn-success btn-sm mr-2" id="approveAllBtn">
                                    <i class="material-icons mr-1" style="font-size: 16px;">done_all</i>
                                    Approve All Today's Requests
                                </button>
                                <button type="button" class="btn btn-info btn-sm mr-2" id="exportPendingBtn">
                                    <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                                    Export Pending List
                                </button>
                                <a href="{{ route('warehouse.outgoing.index', ['status' => 'approved']) }}" class="btn btn-primary btn-sm">
                                    <i class="material-icons mr-1" style="font-size: 16px;">local_shipping</i>
                                    View Ready for Dispatch
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endif
</div>

<!-- Approve Modal -->
<div class="modal fade" id="approveModal" tabindex="-1" role="dialog" aria-labelledby="approveModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="approveModalLabel">
                    <i class="material-icons align-middle mr-1">check_circle</i>
                    Approve Transaction
                </h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to approve transaction <strong id="approveReference"></strong>?</p>
                <p class="text-muted small">This will change the status to "approved" and make it ready for dispatch.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <form id="approveForm" method="POST" style="display: inline;">
                    @csrf
                    <button type="submit" class="btn btn-success">
                        <i class="material-icons mr-1" style="font-size: 16px;">check</i>
                        Approve Transaction
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Handle approve button clicks
        document.querySelectorAll('.approve-btn').forEach(button => {
            button.addEventListener('click', function() {
                const transactionId = this.getAttribute('data-id');
                const reference = this.getAttribute('data-reference');

                document.getElementById('approveReference').textContent = reference;
                document.getElementById('approveForm').action = `/warehouse/outgoing/${transactionId}/approve`;
            });
        });

        // Handle reject button clicks
        document.querySelectorAll('.reject-btn').forEach(button => {
            button.addEventListener('click', function() {
                const transactionId = this.getAttribute('data-id');
                const reference = this.getAttribute('data-reference');

                if (confirm(`Are you sure you want to reject transaction ${reference}?`)) {
                    // Create and submit a reject form
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.action = `/warehouse/outgoing/${transactionId}/reject`;

                    const csrfToken = document.createElement('input');
                    csrfToken.type = 'hidden';
                    csrfToken.name = '_token';
                    csrfToken.value = '{{ csrf_token() }}';
                    form.appendChild(csrfToken);

                    document.body.appendChild(form);
                    form.submit();
                }
            });
        });

        // Handle approve all button
        document.getElementById('approveAllBtn')?.addEventListener('click', function() {
            if (confirm('Are you sure you want to approve all today\'s pending transactions?')) {
                // Implement bulk approval logic
                alert('Bulk approval feature coming soon!');
            }
        });

        // Handle export button
        document.getElementById('exportPendingBtn')?.addEventListener('click', function() {
            // Implement export logic
            alert('Export feature coming soon!');
        });
    });
</script>
@endpush