// Video Loop for Hero and Who We Are sections
document.addEventListener('DOMContentLoaded', function() {
    console.log('Setting up video loops...');
    
    // Function to setup video loop for any section
    function setupVideoLoop() {
        // Hero section video
        const heroSection = document.querySelector('.hero-section');
        if (heroSection) {
            setupSectionVideo(heroSection, 'Hero');
        }
        
        // Who we are section video
        const whoWeAreVideo = document.querySelector('.video-player-section');
        if (whoWeAreVideo) {
            setupSectionVideo(whoWeAreVideo, 'Who We Are');
        }
        
        // Also check for any other video elements on the page
        observeVideoChanges();
    }
    
    // Setup video for a specific section
    function setupSectionVideo(section, sectionName) {
        // Look for video iframe (YouTube embed)
        const videoContainer = section.querySelector('.background-video-embed, .video-wrapper, .video-container');
        
        if (videoContainer) {
            // Check if it's a YouTube video
            const iframe = videoContainer.querySelector('iframe');
            
            if (iframe && iframe.src.includes('youtube.com')) {
                // Update YouTube iframe to include loop parameters
                const currentSrc = iframe.src;
                const videoId = extractYouTubeId(currentSrc);
                
                if (videoId) {
                    // Add loop and playlist parameters for YouTube with mobile optimization
                    const separator = currentSrc.includes('?') ? '&' : '?';
                    const loopParams = `${separator}loop=1&playlist=${videoId}&autoplay=1&mute=1&controls=0&showinfo=0&modestbranding=1&rel=0&playsinline=1`;
                    
                    if (!currentSrc.includes('loop=1')) {
                        iframe.src = currentSrc + loopParams;
                        console.log(`${sectionName} YouTube video loop enabled with mobile optimization`);
                    }
                }
            }
            
            // For HTML5 video elements
            const video = videoContainer.querySelector('video');
            if (video) {
                video.loop = true;
                video.autoplay = true;
                video.muted = true;
                video.playsInline = true;
                
                // Additional mobile optimization
                video.setAttribute('webkit-playsinline', 'true');
                video.setAttribute('playsinline', 'true');
                
                // Force play on mobile devices
                const isMobile = /Android|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
                if (isMobile) {
                    // Attempt to play the video with user interaction fallback
                    const playPromise = video.play();
                    if (playPromise !== undefined) {
                        playPromise.catch(error => {
                            console.log('Mobile video autoplay prevented, will play on user interaction');
                            // Add click listener for manual play
                            document.addEventListener('touchstart', function playOnTouch() {
                                video.play();
                                document.removeEventListener('touchstart', playOnTouch);
                            }, { once: true });
                        });
                    }
                }
                
                console.log(`${sectionName} HTML5 video loop enabled with mobile optimizations`);
            }
        }
    }
    
    // Extract YouTube video ID from URL
    function extractYouTubeId(url) {
        const match = url.match(/(?:youtube\.com\/embed\/|watch\?v=|youtu\.be\/)([^&\?\/]+)/);
        return match ? match[1] : null;
    }
    
    // Observe for dynamically loaded videos
    function observeVideoChanges() {
        const observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.addedNodes.length) {
                    mutation.addedNodes.forEach(function(node) {
                        if (node.nodeType === 1) { // Element node
                            // Check for video elements
                            if (node.tagName === 'VIDEO') {
                                node.loop = true;
                                node.autoplay = true;
                                node.muted = true;
                                node.playsInline = true;
                                
                                // Additional mobile optimization
                                node.setAttribute('webkit-playsinline', 'true');
                                node.setAttribute('playsinline', 'true');
                                
                                // Force play on mobile devices
                                const isMobile = /Android|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
                                if (isMobile) {
                                    const playPromise = node.play();
                                    if (playPromise !== undefined) {
                                        playPromise.catch(error => {
                                            console.log('Mobile video autoplay prevented, will play on user interaction');
                                            document.addEventListener('touchstart', function playOnTouch() {
                                                node.play();
                                                document.removeEventListener('touchstart', playOnTouch);
                                            }, { once: true });
                                        });
                                    }
                                }
                                
                                console.log('Dynamically loaded video set to loop with mobile optimizations');
                            }
                            
                            // Check for YouTube iframes
                            if (node.tagName === 'IFRAME' && node.src.includes('youtube.com')) {
                                const videoId = extractYouTubeId(node.src);
                                if (videoId && !node.src.includes('loop=1')) {
                                    const separator = node.src.includes('?') ? '&' : '?';
                                    node.src += `${separator}loop=1&playlist=${videoId}&autoplay=1&mute=1&playsinline=1`;
                                    console.log('Dynamically loaded YouTube video set to loop with mobile optimization');
                                }
                            }
                            
                            // Check for video elements within added nodes
                            const videos = node.querySelectorAll ? node.querySelectorAll('video') : [];
                            videos.forEach(function(video) {
                                video.loop = true;
                                video.autoplay = true;
                                video.muted = true;
                                video.playsInline = true;
                                
                                // Additional mobile optimization
                                video.setAttribute('webkit-playsinline', 'true');
                                video.setAttribute('playsinline', 'true');
                                
                                // Force play on mobile devices
                                const isMobile = /Android|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
                                if (isMobile) {
                                    const playPromise = video.play();
                                    if (playPromise !== undefined) {
                                        playPromise.catch(error => {
                                            console.log('Mobile nested video autoplay prevented, will play on user interaction');
                                            document.addEventListener('touchstart', function playOnTouch() {
                                                video.play();
                                                document.removeEventListener('touchstart', playOnTouch);
                                            }, { once: true });
                                        });
                                    }
                                }
                            });
                        }
                    });
                }
            });
        });
        
        // Observe the hero section for changes
        const heroSection = document.querySelector('.hero-section');
        if (heroSection) {
            observer.observe(heroSection, {
                childList: true,
                subtree: true
            });
        }
    }
    
    // Initialize on page load
    setupVideoLoop();
    
    // Also run after a delay to catch lazy-loaded videos
    setTimeout(setupVideoLoop, 2000);
    
    // Listen for frontend init events
    if (typeof jQuery !== 'undefined') {
        jQuery(window).on('frontend/init', function() {
            setTimeout(setupVideoLoop, 1000);
        });
    }
    
    console.log('Hero video loop script initialized');
});