// Hero Section Video Background

(function() {
    'use strict';

    // Configuration
    const config = {
        videoId: 'MNzWVa9q9xE',
        startTime: 0,
        mute: true,
        loop: true,
        quality: 'hd720'
    };

    // Initialize when DOM is ready
    document.addEventListener('DOMContentLoaded', initHeroVideo);

    function initHeroVideo() {
        const videoContainer = document.querySelector('.hero-video-bg');
        
        if (!videoContainer) {
            console.log('Hero video container not found');
            return;
        }

        // Enable video on all devices including mobile
        console.log('Initializing hero video for all devices');

        // Add loading state
        const heroSection = document.querySelector('.hero-section-exact');
        if (heroSection) {
            heroSection.classList.add('video-loading');
        }

        // Create YouTube iframe
        createYouTubeEmbed(videoContainer);
    }

    function createYouTubeEmbed(container) {
        // Create loading indicator
        const loadingDiv = document.createElement('div');
        loadingDiv.className = 'hero-loading';
        loadingDiv.textContent = 'Loading video...';
        container.appendChild(loadingDiv);

        // Create iframe
        const iframe = document.createElement('iframe');
        
        // Build YouTube URL with parameters
        const params = new URLSearchParams({
            autoplay: 1,
            mute: config.mute ? 1 : 0,
            loop: config.loop ? 1 : 0,
            playlist: config.videoId, // Required for loop to work
            controls: 0,
            showinfo: 0,
            rel: 0,
            modestbranding: 1,
            iv_load_policy: 3,
            disablekb: 1,
            enablejsapi: 1,
            origin: window.location.origin,
            start: config.startTime,
            vq: config.quality,
            playsinline: 1 // Enable inline playback on mobile
        });

        iframe.src = `https://www.youtube.com/embed/${config.videoId}?${params.toString()}`;
        iframe.frameBorder = '0';
        iframe.allow = 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture';
        iframe.allowFullscreen = false;
        iframe.setAttribute('loading', 'lazy');
        
        // Mobile-specific attributes
        if (window.innerWidth <= 768) {
            iframe.setAttribute('playsinline', 'true');
            iframe.style.pointerEvents = 'none';
        }
        
        // Handle iframe load
        iframe.onload = function() {
            // Remove loading indicator
            if (loadingDiv.parentNode) {
                loadingDiv.remove();
            }
            
            // Remove loading class
            const heroSection = document.querySelector('.hero-section-exact');
            if (heroSection) {
                heroSection.classList.remove('video-loading');
            }
        };
        
        // Handle iframe error for mobile fallback
        iframe.onerror = function() {
            console.log('YouTube iframe failed, trying mobile fallback');
            if (window.innerWidth <= 768) {
                // Try HTML5 video fallback for mobile
                tryMobileFallback(container, loadingDiv);
            }
        };
        
        // Timeout fallback for mobile - if video doesn't load in 5 seconds
        if (window.innerWidth <= 768) {
            setTimeout(function() {
                if (loadingDiv.parentNode) {
                    console.log('Video loading timeout, using mobile fallback');
                    tryMobileFallback(container, loadingDiv);
                }
            }, 5000);
        }

        // Add iframe to container
        container.appendChild(iframe);
    }

    // Mobile fallback function
    function tryMobileFallback(container, loadingDiv) {
        // Remove any existing iframe
        const existingIframe = container.querySelector('iframe');
        if (existingIframe) {
            existingIframe.remove();
        }
        
        // Create mobile-friendly background
        container.style.background = 'linear-gradient(135deg, #038BB9 0%, #1169B5 100%)';
        container.style.backgroundSize = 'cover';
        
        // Remove loading indicator
        if (loadingDiv && loadingDiv.parentNode) {
            loadingDiv.remove();
        }
        
        // Remove loading class
        const heroSection = document.querySelector('.hero-section-exact');
        if (heroSection) {
            heroSection.classList.remove('video-loading');
        }
        
        console.log('Mobile fallback applied - gradient background');
    }

    // Alternative: Use HTML5 video with local file
    function createHTML5Video(container, videoUrl) {
        const video = document.createElement('video');
        video.autoplay = true;
        video.muted = true;
        video.loop = true;
        video.playsInline = true;
        
        // Add source
        const source = document.createElement('source');
        source.src = videoUrl;
        source.type = 'video/mp4';
        
        video.appendChild(source);
        
        // Handle video events
        video.addEventListener('loadeddata', function() {
            const heroSection = document.querySelector('.hero-section-exact');
            if (heroSection) {
                heroSection.classList.remove('video-loading');
            }
        });

        video.addEventListener('error', function() {
            console.error('Video failed to load');
            // Fallback to static background
            const heroSection = document.querySelector('.hero-section-exact');
            if (heroSection) {
                heroSection.classList.remove('video-loading');
            }
        });

        container.appendChild(video);
        
        // Try to play
        video.play().catch(function(error) {
            console.error('Video autoplay failed:', error);
        });
    }

    // Handle window resize
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            const videoContainer = document.querySelector('.hero-video-bg');
            const iframe = videoContainer ? videoContainer.querySelector('iframe') : null;
            
            // Keep video visible on all screen sizes
            if (iframe) {
                iframe.style.display = 'block';
            } else if (!iframe && videoContainer) {
                // Reinitialize video if not present
                initHeroVideo();
            }
        }, 250);
    });

    // Optimize performance - pause video when not visible
    if ('IntersectionObserver' in window) {
        const observer = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                const iframe = entry.target.querySelector('iframe');
                if (iframe) {
                    if (entry.isIntersecting) {
                        // Video is visible - could trigger play via postMessage
                        iframe.contentWindow.postMessage('{"event":"command","func":"playVideo","args":""}', '*');
                    } else {
                        // Video is not visible - could trigger pause via postMessage
                        iframe.contentWindow.postMessage('{"event":"command","func":"pauseVideo","args":""}', '*');
                    }
                }
            });
        }, {
            threshold: 0.1
        });

        // Observe hero section
        const heroSection = document.querySelector('.hero-section-exact');
        if (heroSection) {
            observer.observe(heroSection);
        }
    }

    // Export for external use
    window.HeroVideo = {
        init: initHeroVideo,
        config: config,
        updateVideo: function(newVideoId) {
            config.videoId = newVideoId;
            const container = document.querySelector('.hero-video-bg');
            if (container) {
                container.innerHTML = '';
                initHeroVideo();
            }
        }
    };
})();