// Dynamic Video Manager - Manages videos from backend settings
class DynamicVideoManager {
    constructor() {
        this.videoConfigs = {};
        this.loadedVideos = new Set();
        this.preloadedVideos = new Set();
        this.init();
    }

    async init() {
        try {
            console.log('🎬 Dynamic Video Manager initializing...');
            await this.loadVideoConfigs();
            console.log('📍 Current page path:', window.location.pathname);
            this.initializeVideos();
            this.setupPreloading();
            this.initializePreloader();
            console.log('✅ Dynamic Video Manager initialized successfully');
        } catch (error) {
            console.error('❌ Dynamic Video Manager initialization failed:', error);
            this.fallbackToDefaultVideos();
        }
    }

    initializePreloader() {
        // Load the video preloader if available
        if (typeof window.videoPreloader === 'undefined') {
            // Load preloader script dynamically
            const script = document.createElement('script');
            script.src = '/assets/js/video-preloader.js';
            script.onload = () => {
                this.startIntelligentPreloading();
            };
            document.head.appendChild(script);
        } else {
            this.startIntelligentPreloading();
        }
    }

    startIntelligentPreloading() {
        if (window.videoPreloader) {
            // Preload all configured videos
            Object.values(this.videoConfigs).forEach(config => {
                if (config.preloadAt) {
                    window.videoPreloader.preloadVideo(config, 'normal')
                        .then(() => {
                            console.log(`✅ Video preloaded: ${config.key}`);
                        })
                        .catch(error => {
                            console.warn(`⚠️ Failed to preload ${config.key}:`, error);
                        });
                }
            });
        }
    }

    async loadVideoConfigs() {
        const response = await fetch('/api/video-configs');
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        this.videoConfigs = await response.json();
        console.log('Video configs loaded:', this.videoConfigs);
    }

    initializeVideos() {
        console.log('🎥 Initializing videos...');
        
        // Initialize hero video
        if (this.videoConfigs.hero_video) {
            console.log('🏠 Initializing hero video');
            this.initializeHeroVideo();
        }

        // Initialize page-specific videos based on current page
        const currentPath = window.location.pathname;
        console.log('📍 Current path for video selection:', currentPath);
        
        if (currentPath === '/' || currentPath === '/home') {
            // Home page - initialize who we are video
            if (this.videoConfigs.who_we_are_video) {
                console.log('🏠 Home page detected - initializing who we are video');
                this.initializeWhoWeAreVideo();
            } else {
                console.log('⚠️ No who_we_are_video config found');
            }
        } else if (currentPath === '/about' || currentPath.includes('/about')) {
            // About page - initialize about video
            if (this.videoConfigs.about_video) {
                console.log('ℹ️ About page detected - initializing about video');
                this.initializeAboutVideo();
            } else {
                console.log('⚠️ No about_video config found');
            }
        } else {
            console.log('❓ Unknown page - no page-specific video initialization');
        }
    }

    initializeHeroVideo() {
        const config = this.videoConfigs.hero_video;
        const videoContainer = document.querySelector('.hero-video-bg');
        
        if (!videoContainer || window.innerWidth <= 768) {
            return; // Skip on mobile or if container not found
        }

        // Clear existing content
        videoContainer.innerHTML = '';
        
        // Add loading state
        const heroSection = document.querySelector('.hero-section-exact');
        if (heroSection) {
            heroSection.classList.add('video-loading');
        }

        this.createVideoElement(videoContainer, config, () => {
            if (heroSection) {
                heroSection.classList.remove('video-loading');
            }
        });

        this.loadedVideos.add('hero_video');
    }

    initializeWhoWeAreVideo() {
        const config = this.videoConfigs.who_we_are_video;
        console.log('🎬 Who We Are video config:', config);
        
        // Look for the who we are video element
        const videoElement = document.querySelector('.ufg-video');
        console.log('🔍 Found .ufg-video element:', videoElement ? 'Yes' : 'No');
        
        if (videoElement && config.embedUrl) {
            console.log('🔄 Current src:', videoElement.src);
            console.log('🔄 New src:', config.embedUrl);
            
            // Update the src with the correct backend configuration
            videoElement.src = config.embedUrl;
            console.log('✅ Who We Are video updated with backend config');
            this.loadedVideos.add('who_we_are_video');
        } else {
            console.log('❌ Cannot update Who We Are video - missing element or embedUrl');
        }
    }

    initializeAboutVideo() {
        const config = this.videoConfigs.about_video;
        console.log('📰 About video config:', config);
        
        // Look for about page video element
        const videoElement = document.querySelector('.ufg-video');
        console.log('🔍 Found .ufg-video element:', videoElement ? 'Yes' : 'No');
        
        if (videoElement && config.embedUrl) {
            console.log('🔄 Current src:', videoElement.src);
            console.log('🔄 New src:', config.embedUrl);
            
            // Update the src with the correct backend configuration
            videoElement.src = config.embedUrl;
            console.log('✅ About video updated with backend config');
            this.loadedVideos.add('about_video');
        } else {
            console.log('❌ Cannot update About video - missing element or embedUrl');
        }
    }

    createVideoElement(container, config, onLoadCallback) {
        if (config.type === 'youtube' && config.videoId) {
            this.createYouTubeEmbed(container, config, onLoadCallback);
        } else if (config.type === 'vimeo' && config.videoId) {
            this.createVimeoEmbed(container, config, onLoadCallback);
        } else if (config.type === 'mp4' || config.type === 'uploaded' || config.isUploaded) {
            this.createHTML5Video(container, config, onLoadCallback);
        }
    }

    createYouTubeEmbed(container, config, onLoadCallback) {
        // Create loading indicator
        const loadingDiv = document.createElement('div');
        loadingDiv.className = 'hero-loading';
        loadingDiv.textContent = 'Loading video...';
        container.appendChild(loadingDiv);

        // Create iframe
        const iframe = document.createElement('iframe');
        
        // Use the pre-built embed URL from backend or construct it
        let embedUrl = config.embedUrl;
        if (!embedUrl) {
            const params = new URLSearchParams(config.settings || {});
            embedUrl = `https://www.youtube.com/embed/${config.videoId}?${params.toString()}`;
        }

        iframe.src = embedUrl;
        iframe.frameBorder = '0';
        iframe.allow = 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture';
        iframe.allowFullscreen = false;
        iframe.setAttribute('loading', 'lazy');
        
        // Handle iframe load
        iframe.onload = function() {
            if (loadingDiv.parentNode) {
                loadingDiv.remove();
            }
            if (onLoadCallback) {
                onLoadCallback();
            }
        };

        iframe.onerror = function() {
            console.error('Failed to load YouTube video:', config.videoId);
            if (loadingDiv.parentNode) {
                loadingDiv.remove();
            }
            if (onLoadCallback) {
                onLoadCallback();
            }
        };

        container.appendChild(iframe);
    }

    createVimeoEmbed(container, config, onLoadCallback) {
        // Create loading indicator
        const loadingDiv = document.createElement('div');
        loadingDiv.className = 'hero-loading';
        loadingDiv.textContent = 'Loading video...';
        container.appendChild(loadingDiv);

        // Create iframe
        const iframe = document.createElement('iframe');
        
        // Use the pre-built embed URL from backend or construct it
        let embedUrl = config.embedUrl;
        if (!embedUrl && config.videoId) {
            const params = new URLSearchParams(config.settings || {});
            embedUrl = `https://player.vimeo.com/video/${config.videoId}?${params.toString()}`;
        }

        iframe.src = embedUrl;
        iframe.frameBorder = '0';
        iframe.allow = 'autoplay; fullscreen; picture-in-picture';
        iframe.allowFullscreen = false;
        iframe.setAttribute('loading', 'lazy');
        
        // Handle iframe load
        iframe.onload = function() {
            if (loadingDiv.parentNode) {
                loadingDiv.remove();
            }
            if (onLoadCallback) {
                onLoadCallback();
            }
        };

        iframe.onerror = function() {
            console.error('Failed to load Vimeo video:', config.videoId);
            if (loadingDiv.parentNode) {
                loadingDiv.remove();
            }
            if (onLoadCallback) {
                onLoadCallback();
            }
        };

        container.appendChild(iframe);
    }

    createHTML5Video(container, config, onLoadCallback) {
        // Create loading indicator
        const loadingDiv = document.createElement('div');
        loadingDiv.className = 'hero-loading';
        loadingDiv.textContent = 'Loading video...';
        container.appendChild(loadingDiv);

        const video = document.createElement('video');
        const settings = config.settings || {};
        
        video.autoplay = settings.autoplay || false;
        video.muted = settings.mute || false;
        video.loop = settings.loop || false;
        video.controls = settings.controls || false;
        video.playsInline = true;
        video.style.width = '100%';
        video.style.height = '100%';
        video.style.objectFit = 'cover';
        
        // Add source - prioritize videoUrl from backend config
        const source = document.createElement('source');
        source.src = config.videoUrl || config.embedUrl;
        
        // Set appropriate MIME type based on file extension or config
        if (config.isUploaded || config.type === 'uploaded') {
            const extension = source.src.split('.').pop().toLowerCase();
            switch (extension) {
                case 'mp4':
                    source.type = 'video/mp4';
                    break;
                case 'webm':
                    source.type = 'video/webm';
                    break;
                case 'mov':
                    source.type = 'video/quicktime';
                    break;
                case 'avi':
                    source.type = 'video/x-msvideo';
                    break;
                default:
                    source.type = 'video/mp4';
            }
        } else {
            source.type = 'video/mp4';
        }
        
        video.appendChild(source);
        
        // Handle video events
        video.addEventListener('loadeddata', function() {
            if (loadingDiv.parentNode) {
                loadingDiv.remove();
            }
            if (onLoadCallback) {
                onLoadCallback();
            }
        });

        video.addEventListener('canplay', function() {
            if (loadingDiv.parentNode) {
                loadingDiv.remove();
            }
        });

        video.addEventListener('error', function() {
            console.error('Video failed to load:', source.src);
            if (loadingDiv.parentNode) {
                loadingDiv.remove();
            }
            if (onLoadCallback) {
                onLoadCallback();
            }
        });

        container.appendChild(video);
        
        // Try to play
        if (video.autoplay) {
            video.play().catch(function(error) {
                console.error('Video autoplay failed:', error);
                // On autoplay failure, show controls
                video.controls = true;
            });
        }
    }

    setupPreloading() {
        // Preload videos that are marked for preloading
        Object.keys(this.videoConfigs).forEach(key => {
            const config = this.videoConfigs[key];
            if (config.preloadAt && !this.preloadedVideos.has(key)) {
                this.preloadVideo(key, config);
            }
        });

        // Set up intersection observer for lazy preloading
        this.setupIntersectionObserver();
    }

    preloadVideo(key, config) {
        if (config.type === 'youtube' && config.videoId) {
            // Preload YouTube thumbnail
            const img = new Image();
            img.src = config.thumbnailUrl || `https://img.youtube.com/vi/${config.videoId}/maxresdefault.jpg`;
            img.onload = () => {
                console.log(`Preloaded thumbnail for ${key}`);
                this.preloadedVideos.add(key);
            };

            // Preload YouTube iframe API if not already loaded
            if (typeof YT === 'undefined') {
                const tag = document.createElement('script');
                tag.src = 'https://www.youtube.com/iframe_api';
                const firstScriptTag = document.getElementsByTagName('script')[0];
                firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
            }
        } else if (config.type === 'vimeo' && config.videoId) {
            // Preload Vimeo thumbnail if available
            if (config.thumbnailUrl) {
                const img = new Image();
                img.src = config.thumbnailUrl;
                img.onload = () => {
                    console.log(`Preloaded Vimeo thumbnail for ${key}`);
                    this.preloadedVideos.add(key);
                };
            }
        } else if (config.type === 'mp4' || config.type === 'uploaded' || config.isUploaded) {
            // Preload uploaded/MP4 video
            const video = document.createElement('video');
            video.preload = 'metadata';
            video.src = config.videoUrl;
            
            // Preload thumbnail if available
            if (config.thumbnailUrl) {
                const img = new Image();
                img.src = config.thumbnailUrl;
                img.onload = () => {
                    console.log(`Preloaded thumbnail for ${key}`);
                };
            }
            
            video.onloadedmetadata = () => {
                console.log(`Preloaded metadata for ${key}`);
                this.preloadedVideos.add(key);
            };
            
            video.onerror = () => {
                console.warn(`Failed to preload video metadata for ${key}`);
            };
        }
    }

    setupIntersectionObserver() {
        if ('IntersectionObserver' in window) {
            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const iframe = entry.target.querySelector('iframe');
                        if (iframe && iframe.src.includes('youtube.com')) {
                            // Video is visible - play if autoplay enabled
                            try {
                                iframe.contentWindow.postMessage('{"event":"command","func":"playVideo","args":""}', '*');
                            } catch (e) {
                                // Ignore cross-origin errors
                            }
                        }
                    } else {
                        const iframe = entry.target.querySelector('iframe');
                        if (iframe && iframe.src.includes('youtube.com')) {
                            // Video is not visible - pause to save bandwidth
                            try {
                                iframe.contentWindow.postMessage('{"event":"command","func":"pauseVideo","args":""}', '*');
                            } catch (e) {
                                // Ignore cross-origin errors
                            }
                        }
                    }
                });
            }, {
                threshold: 0.1
            });

            // Observe video containers
            const heroSection = document.querySelector('.hero-section-exact');
            if (heroSection) {
                observer.observe(heroSection);
            }

            const videoShowcase = document.querySelector('.modern-video-showcase');
            if (videoShowcase) {
                observer.observe(videoShowcase);
            }
        }
    }

    fallbackToDefaultVideos() {
        console.warn('Falling back to default video configuration');
        
        // Fallback configuration
        this.videoConfigs = {
            hero_video: {
                type: 'youtube',
                videoId: 'MNzWVa9q9xE',
                settings: {
                    autoplay: 1,
                    mute: 1,
                    loop: 1,
                    controls: 0
                }
            },
            who_we_are_video: {
                type: 'youtube',
                videoId: 'MNzWVa9q9xE',
                settings: {
                    autoplay: 1,
                    controls: 1
                }
            },
            about_video: {
                type: 'youtube',
                videoId: 'MNzWVa9q9xE',
                settings: {
                    autoplay: 1,
                    controls: 1
                }
            }
        };

        this.initializeVideos();
    }

    // Public methods for manual control
    updateVideo(key, newConfig) {
        this.videoConfigs[key] = newConfig;
        
        // Re-initialize specific video
        switch (key) {
            case 'hero_video':
                this.initializeHeroVideo();
                break;
            case 'who_we_are_video':
                this.initializeWhoWeAreVideo();
                break;
            case 'about_video':
                this.initializeAboutVideo();
                break;
        }
    }

    getVideoConfig(key) {
        return this.videoConfigs[key];
    }

    isVideoLoaded(key) {
        return this.loadedVideos.has(key);
    }

    isVideoPreloaded(key) {
        return this.preloadedVideos.has(key);
    }

    // Handle window resize for mobile optimization
    handleResize() {
        if (window.innerWidth <= 768) {
            // Hide videos on mobile
            const heroVideo = document.querySelector('.hero-video-bg iframe');
            if (heroVideo) {
                heroVideo.style.display = 'none';
            }
        } else {
            // Show videos on desktop
            const heroVideo = document.querySelector('.hero-video-bg iframe');
            if (heroVideo) {
                heroVideo.style.display = 'block';
            } else if (this.videoConfigs.hero_video) {
                // Re-initialize if video was removed
                this.initializeHeroVideo();
            }
        }
    }
}

// Initialize the dynamic video manager
let videoManager;

// Wait for DOM to be ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        videoManager = new DynamicVideoManager();
    });
} else {
    videoManager = new DynamicVideoManager();
}

// Handle window resize
let resizeTimer;
window.addEventListener('resize', function() {
    clearTimeout(resizeTimer);
    resizeTimer = setTimeout(function() {
        if (videoManager) {
            videoManager.handleResize();
        }
    }, 250);
});

// Export for global access
window.DynamicVideoManager = DynamicVideoManager;
window.videoManager = videoManager;