// Custom Video Controls for YouTube Video
(function() {
    'use strict';

    let player;
    let isPlaying = false;
    let duration = 0;
    let currentTime = 0;
    let isMuted = true;

    // Initialize when DOM is ready
    document.addEventListener('DOMContentLoaded', function() {
        initCustomControls();
        loadYouTubeAPI();
    });

    function loadYouTubeAPI() {
        // Load YouTube IFrame API if not already loaded
        if (typeof YT === 'undefined') {
            const tag = document.createElement('script');
            tag.src = 'https://www.youtube.com/iframe_api';
            const firstScriptTag = document.getElementsByTagName('script')[0];
            firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
        }
        
        // YouTube API ready callback
        window.onYouTubeIframeAPIReady = function() {
            initYouTubePlayer();
        };
    }

    function initYouTubePlayer() {
        const iframe = document.getElementById('widget4');
        if (iframe) {
            player = new YT.Player('widget4', {
                events: {
                    'onReady': onPlayerReady,
                    'onStateChange': onPlayerStateChange
                }
            });
        }
    }

    function onPlayerReady(event) {
        console.log('YouTube player ready');
        duration = player.getDuration();
        updateTimeDisplay();
        
        // Start playing automatically (muted)
        player.mute();
        player.playVideo();
        isPlaying = true;
        isMuted = true;
        updatePlayButton();
        updateVolumeButton();
    }

    function onPlayerStateChange(event) {
        if (event.data === YT.PlayerState.PLAYING) {
            isPlaying = true;
            startProgressUpdate();
        } else if (event.data === YT.PlayerState.PAUSED) {
            isPlaying = false;
            stopProgressUpdate();
        }
        updatePlayButton();
    }

    function initCustomControls() {
        const playBtn = document.querySelector('.video-play-btn');
        const volumeBtn = document.querySelector('.video-volume-btn');
        const fullscreenBtn = document.querySelector('.video-fullscreen-btn');
        const progressBar = document.querySelector('.video-progress');

        if (playBtn) {
            playBtn.addEventListener('click', togglePlay);
            playBtn.innerHTML = '⏸️'; // Pause icon (starts playing)
        }

        if (volumeBtn) {
            volumeBtn.addEventListener('click', toggleMute);
            volumeBtn.innerHTML = '🔇'; // Muted icon (starts muted)
        }

        if (fullscreenBtn) {
            fullscreenBtn.addEventListener('click', toggleFullscreen);
            fullscreenBtn.innerHTML = '⛶';
        }

        if (progressBar) {
            progressBar.addEventListener('click', seekVideo);
        }
    }

    function togglePlay() {
        if (!player) return;

        if (isPlaying) {
            player.pauseVideo();
        } else {
            player.playVideo();
        }
    }

    function updatePlayButton() {
        const playBtn = document.querySelector('.video-play-btn');
        if (playBtn) {
            playBtn.innerHTML = isPlaying ? '⏸️' : '▶️';
        }
    }

    function toggleMute() {
        if (!player) return;

        if (isMuted) {
            player.unMute();
            isMuted = false;
        } else {
            player.mute();
            isMuted = true;
        }
        updateVolumeButton();
    }

    function updateVolumeButton() {
        const volumeBtn = document.querySelector('.video-volume-btn');
        if (volumeBtn) {
            volumeBtn.innerHTML = isMuted ? '🔇' : '🔊';
        }
    }

    function toggleFullscreen() {
        const iframe = document.getElementById('widget4');
        if (!iframe) return;

        if (document.fullscreenElement) {
            document.exitFullscreen();
        } else {
            iframe.requestFullscreen().catch(err => {
                console.error('Error attempting to enable fullscreen:', err);
            });
        }
    }

    function seekVideo(event) {
        if (!player || !duration) return;

        const progressBar = event.currentTarget;
        const rect = progressBar.getBoundingClientRect();
        const clickX = event.clientX - rect.left;
        const percentage = clickX / rect.width;
        const seekTime = duration * percentage;

        player.seekTo(seekTime, true);
    }

    let progressInterval;

    function startProgressUpdate() {
        stopProgressUpdate();
        progressInterval = setInterval(updateProgress, 100);
    }

    function stopProgressUpdate() {
        if (progressInterval) {
            clearInterval(progressInterval);
            progressInterval = null;
        }
    }

    function updateProgress() {
        if (!player) return;

        try {
            currentTime = player.getCurrentTime();
            duration = player.getDuration();
            
            const percentage = duration > 0 ? (currentTime / duration) * 100 : 0;
            
            const progressBar = document.querySelector('.video-progress-bar');
            if (progressBar) {
                progressBar.style.width = percentage + '%';
            }

            updateTimeDisplay();
        } catch (error) {
            console.error('Error updating progress:', error);
        }
    }

    function updateTimeDisplay() {
        const timeDisplay = document.querySelector('.video-time');
        if (timeDisplay) {
            const current = formatTime(currentTime);
            const total = formatTime(duration);
            timeDisplay.textContent = `${current} / ${total}`;
        }
    }

    function formatTime(seconds) {
        if (!seconds || isNaN(seconds)) return '0:00';
        
        const minutes = Math.floor(seconds / 60);
        const remainingSeconds = Math.floor(seconds % 60);
        return `${minutes}:${remainingSeconds.toString().padStart(2, '0')}`;
    }

    // Export for external use
    window.CustomVideoControls = {
        togglePlay: togglePlay,
        toggleMute: toggleMute,
        toggleFullscreen: toggleFullscreen
    };

})();