<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use App\Models\User;
use Illuminate\Database\Seeder;

class SuperAdminSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create comprehensive system permissions
        $systemPermissions = [
            // User Management
            [
                'name' => 'users.view',
                'display_name' => 'View Users',
                'description' => 'Can view user list and details',
                'group' => 'user_management'
            ],
            [
                'name' => 'users.create',
                'display_name' => 'Create Users',
                'description' => 'Can create new users',
                'group' => 'user_management'
            ],
            [
                'name' => 'users.edit',
                'display_name' => 'Edit Users',
                'description' => 'Can edit user information',
                'group' => 'user_management'
            ],
            [
                'name' => 'users.delete',
                'display_name' => 'Delete Users',
                'description' => 'Can delete users',
                'group' => 'user_management'
            ],

            // Role Management
            [
                'name' => 'roles.view',
                'display_name' => 'View Roles',
                'description' => 'Can view roles and permissions',
                'group' => 'role_management'
            ],
            [
                'name' => 'roles.create',
                'display_name' => 'Create Roles',
                'description' => 'Can create new roles',
                'group' => 'role_management'
            ],
            [
                'name' => 'roles.edit',
                'display_name' => 'Edit Roles',
                'description' => 'Can edit roles and assign permissions',
                'group' => 'role_management'
            ],
            [
                'name' => 'roles.delete',
                'display_name' => 'Delete Roles',
                'description' => 'Can delete roles',
                'group' => 'role_management'
            ],

            // System Settings
            [
                'name' => 'settings.view',
                'display_name' => 'View System Settings',
                'description' => 'Can view system settings',
                'group' => 'system_settings'
            ],
            [
                'name' => 'settings.manage',
                'display_name' => 'Manage System Settings',
                'description' => 'Can modify system settings',
                'group' => 'system_settings'
            ],

            // Department Management
            [
                'name' => 'departments.view',
                'display_name' => 'View Departments',
                'description' => 'Can view departments',
                'group' => 'department_management'
            ],
            [
                'name' => 'departments.create',
                'display_name' => 'Create Departments',
                'description' => 'Can create new departments',
                'group' => 'department_management'
            ],
            [
                'name' => 'departments.edit',
                'display_name' => 'Edit Departments',
                'description' => 'Can edit department information',
                'group' => 'department_management'
            ],
            [
                'name' => 'departments.delete',
                'display_name' => 'Delete Departments',
                'description' => 'Can delete departments',
                'group' => 'department_management'
            ],

            // Activity Logs
            [
                'name' => 'activity_logs.view',
                'display_name' => 'View Activity Logs',
                'description' => 'Can view user activity logs',
                'group' => 'system_monitoring'
            ],

            // Site Management
            [
                'name' => 'site.manage',
                'display_name' => 'Manage Site',
                'description' => 'Can manage site configuration',
                'group' => 'site_management'
            ],

            // Admin Dashboard
            [
                'name' => 'admin.dashboard.view',
                'display_name' => 'View Admin Dashboard',
                'description' => 'Can access admin dashboard',
                'group' => 'admin_dashboard'
            ],

            // System Administration
            [
                'name' => 'system.admin',
                'display_name' => 'System Administrator',
                'description' => 'Full system administration access',
                'group' => 'system_admin'
            ],
        ];

        // Create permissions
        foreach ($systemPermissions as $permission) {
            Permission::firstOrCreate(
                ['name' => $permission['name']],
                $permission
            );
        }

        // Create Super Admin role
        $superAdminRole = Role::firstOrCreate(
            ['name' => 'super_admin'],
            [
                'display_name' => 'Super Administrator',
                'description' => 'Full system access including user management, settings, and all warehouse operations',
                'is_active' => true
            ]
        );

        // Get all permissions (warehouse + system)
        $allPermissions = Permission::all()->pluck('id')->toArray();

        // Assign all permissions to super admin role
        $superAdminRole->permissions()->sync($allPermissions);

        // Update the KCG admin user to super admin role
        $adminUser = User::where('email', 'admin@kcg.com')->first();
        if ($adminUser) {
            $adminUser->update(['role_id' => $superAdminRole->id]);
            $this->command->info('KCG admin user updated to Super Administrator role!');
        }

        // Also create a regular admin role (less than super admin)
        $adminRole = Role::firstOrCreate(
            ['name' => 'admin'],
            [
                'display_name' => 'Administrator',
                'description' => 'System administrator with most privileges except user management',
                'is_active' => true
            ]
        );

        // Assign most permissions to admin role (exclude some sensitive ones)
        $adminPermissions = Permission::whereNotIn('name', [
            'users.delete',
            'roles.delete',
            'system.admin'
        ])->pluck('id')->toArray();

        $adminRole->permissions()->sync($adminPermissions);

        $this->command->info('Super Admin role created with all system permissions!');
        $this->command->info('Admin role created with most system permissions!');
        $this->command->info('');
        $this->command->info('Super Admin permissions include:');
        $this->command->info('- Full User Management');
        $this->command->info('- Complete Role & Permission Management');
        $this->command->info('- System Settings Configuration');
        $this->command->info('- Department Management');
        $this->command->info('- Activity Log Access');
        $this->command->info('- Site Management');
        $this->command->info('- All Warehouse Operations');
        $this->command->info('- System Administration');
    }
}