<?php

namespace App\Models\Warehouse;

use App\Models\Department;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class IncomingTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'transaction_date',
        'transaction_type',
        'supplier_id',
        'project_id',
        'project_number',
        'type_of_goods',
        'lpo_number',
        'delivery_note_number',
        'division_id',
        'total_amount',
        'received_by',
        'received_by_name',
        'status',
        'notes',
        'delivery_note_path'
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'total_amount' => 'decimal:2',
        'status' => 'string',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($transaction) {
            // Recalculate total amount from items
            if ($transaction->exists && $transaction->items()->exists()) {
                $transaction->total_amount = $transaction->items()->sum('total_price');
            }
        });
    }

    /**
     * Get the supplier for the transaction
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * Get the division for the transaction
     */
    public function division(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'division_id');
    }

    /**
     * Get the project for the transaction
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    /**
     * Get the user who received the goods
     */
    public function receiver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'received_by');
    }

    /**
     * Get all items in this transaction
     */
    public function items(): HasMany
    {
        return $this->hasMany(IncomingItem::class);
    }

    /**
     * Get stock movements created by this transaction
     */
    public function stockMovements()
    {
        return $this->morphMany(StockMovement::class, 'reference');
    }

    /**
     * Scope for pending transactions
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for completed transactions
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope for transactions in date range
     */
    public function scopeDateBetween($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_date', [$startDate, $endDate]);
    }

    /**
     * Complete the transaction and update inventory
     */
    public function complete()
    {
        $this->status = 'completed';
        $this->save();

        // Update inventory for each item
        foreach ($this->items as $item) {
            $this->updateInventory($item);
            $this->createStockMovement($item);
        }
    }

    /**
     * Update inventory for an item
     */
    protected function updateInventory($incomingItem)
    {
        $inventory = Inventory::firstOrNew([
            'item_id' => $incomingItem->item_id,
            'division_id' => $this->division_id,
            'batch_number' => $incomingItem->batch_number,
        ]);

        $inventory->quantity_available += $incomingItem->quantity_received;
        $inventory->unit_price = $incomingItem->unit_price;
        $inventory->production_date = $incomingItem->production_date;
        $inventory->expiry_date = $incomingItem->expiry_date;
        $inventory->save();
    }

    /**
     * Create stock movement record
     */
    protected function createStockMovement($incomingItem)
    {
        StockMovement::create([
            'item_id' => $incomingItem->item_id,
            'movement_type' => 'in',
            'reference_type' => 'incoming',
            'reference_id' => $this->id,
            'quantity_before' => 0, // This should be calculated
            'quantity_moved' => $incomingItem->quantity_received,
            'quantity_after' => 0, // This should be calculated
            'division_id' => $this->division_id,
            'user_id' => $this->received_by,
            'notes' => "Incoming from supplier: {$this->supplier->supplier_name}"
        ]);
    }

    /**
     * Get transaction reference number
     */
    public function getReferenceNumberAttribute()
    {
        return $this->delivery_note_number ?: $this->lpo_number ?: 'IN-' . str_pad($this->id, 6, '0', STR_PAD_LEFT);
    }
}