<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\UserActivity;
use Illuminate\Http\Request;

class ActivityLogController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->hasPermission('view_activity_logs')) {
                abort(403, 'Unauthorized access to activity logs');
            }
            return $next($request);
        });
    }

    /**
     * Display activity logs
     */
    public function index(Request $request)
    {
        $query = UserActivity::with('user')->latest();

        // Filter by user
        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Filter by action
        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        // Filter by date range
        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->from_date);
        }
        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->to_date);
        }

        // Search in description
        if ($request->filled('search')) {
            $query->where('description', 'like', '%' . $request->search . '%');
        }

        $activities = $query->paginate(50);
        
        // Get unique actions for filter dropdown
        $actions = UserActivity::select('action')->distinct()->pluck('action');
        
        // Get users who have activities
        $users = \App\Models\User::whereIn('id', UserActivity::select('user_id')->distinct())->get();

        return view('admin.activity-logs.index', compact('activities', 'actions', 'users'));
    }

    /**
     * Clear old activity logs
     */
    public function clear(Request $request)
    {
        if (!auth()->user()->hasPermission('manage_activity_logs')) {
            abort(403, 'Unauthorized to manage activity logs');
        }

        $validated = $request->validate([
            'days' => 'required|integer|min:30'
        ]);

        $date = now()->subDays($validated['days']);
        $count = UserActivity::where('created_at', '<', $date)->count();
        
        UserActivity::where('created_at', '<', $date)->delete();

        return redirect()->route('admin.activity-logs')
            ->with('success', "Deleted {$count} activity logs older than {$validated['days']} days");
    }
}