<?php
require __DIR__ . '/vendor/autoload.php';
$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(\Illuminate\Contracts\Http\Kernel::class);
$response = $kernel->handle(
    $request = \Illuminate\Http\Request::capture()
);

use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\OutgoingFulfillment;
use App\Models\Warehouse\ProjectInventory;

// Get the original transaction
$transaction = OutgoingTransaction::with('items')->find(1);

echo "Testing project inventory update for transaction " . $transaction->id . "\n";
echo "Project ID: " . $transaction->project_id . "\n\n";

// Get all fulfillment records for this transaction
$fulfillments = OutgoingFulfillment::where('outgoing_transaction_id', 1)->get();

echo "Current fulfillments:\n";
foreach ($fulfillments as $fulfillment) {
    echo "- Item " . $fulfillment->item_id . ": " . $fulfillment->quantity_fulfilled . " (" . $fulfillment->fulfillment_type . ")\n";
}

// Group fulfillments by item and fulfillment type
$inventoryItems = [];
$deliveryItems = [];
$transferItems = [];

foreach ($fulfillments as $fulfillment) {
    switch ($fulfillment->fulfillment_type) {
        case 'inventory':
            $inventoryItems[$fulfillment->item_id] = ($inventoryItems[$fulfillment->item_id] ?? 0) + $fulfillment->quantity_fulfilled;
            break;
        case 'direct_delivery':
            $deliveryItems[$fulfillment->item_id] = ($deliveryItems[$fulfillment->item_id] ?? 0) + $fulfillment->quantity_fulfilled;
            break;
        case 'project_transfer':
            $transferItems[$fulfillment->item_id] = ($transferItems[$fulfillment->item_id] ?? 0) + $fulfillment->quantity_fulfilled;
            break;
    }
}

echo "\nGrouped by type:\n";
echo "Inventory: " . json_encode($inventoryItems) . "\n";
echo "Delivery: " . json_encode($deliveryItems) . "\n";
echo "Transfer: " . json_encode($transferItems) . "\n";

echo "\nProject Inventory BEFORE update:\n";
$beforeInventory = ProjectInventory::where('project_id', $transaction->project_id)->get();
foreach ($beforeInventory as $inv) {
    echo "Item " . $inv->item_id . ": " . $inv->quantity_available . "\n";
}

// Simulate the update method
$allFulfilledItems = [];

// Add inventory dispatch quantities
foreach ($inventoryItems as $itemId => $quantity) {
    $allFulfilledItems[$itemId] = ($allFulfilledItems[$itemId] ?? 0) + $quantity;
}

// Add direct delivery quantities
foreach ($deliveryItems as $itemId => $quantity) {
    $allFulfilledItems[$itemId] = ($allFulfilledItems[$itemId] ?? 0) + $quantity;
}

// Add project transfer quantities
foreach ($transferItems as $itemId => $quantity) {
    $allFulfilledItems[$itemId] = ($allFulfilledItems[$itemId] ?? 0) + $quantity;
}

echo "\nTotal fulfilled per item: " . json_encode($allFulfilledItems) . "\n";

// Reset project inventory to original state for proper testing
ProjectInventory::where('project_id', $transaction->project_id)->delete();
echo "\nReset project inventory to simulate clean state\n";

// Apply the update logic
foreach ($allFulfilledItems as $itemId => $totalFulfilled) {
    if ($totalFulfilled > 0) {
        $projectInventory = ProjectInventory::where('project_id', $transaction->project_id)
            ->where('item_id', $itemId)
            ->first();

        if ($projectInventory) {
            $projectInventory->updateQuantity($totalFulfilled, 'add');
            echo "Updated existing record for item $itemId: added $totalFulfilled\n";
        } else {
            $originalItem = $transaction->items()->where('item_id', $itemId)->first();
            $unitPrice = $originalItem->unit_price ?? 0;

            ProjectInventory::create([
                'project_id' => $transaction->project_id,
                'item_id' => $itemId,
                'quantity_available' => $totalFulfilled,
                'allocated_quantity' => 0,
                'unit_price' => $unitPrice,
                'total_value' => $totalFulfilled * $unitPrice
            ]);
            echo "Created new record for item $itemId: $totalFulfilled units at $unitPrice each\n";
        }
    }
}

echo "\nProject Inventory AFTER update:\n";
$afterInventory = ProjectInventory::where('project_id', $transaction->project_id)->get();
foreach ($afterInventory as $inv) {
    echo "Item " . $inv->item_id . ": " . $inv->quantity_available . " (value: " . $inv->total_value . ")\n";
}