@extends('layouts.admin-simple')

@section('title', 'Material Transfer Issues')

@push('styles')
<style>
/* Adjust Select2 dropdown widths for better proportions */
.select2-container {
    width: 100% !important;
}

/* Make dropdowns auto-width based on content */
.select2-dropdown {
    min-width: 200px;
    max-width: 400px;
    width: auto !important;
}

/* Ensure the selection box doesn't get too wide */
.select2-container .select2-selection--single {
    min-width: 200px;
    max-width: 100%;
}

/* Allow dropdown to expand based on content */
.select2-container--open .select2-dropdown--below {
    width: auto !important;
    min-width: 200px;
}
</style>
@endpush

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="card-title mb-0">
                        <i class="material-icons mr-2">local_shipping</i>
                        Material Transfer Issues
                    </h4>
                </div>

                <div class="card-body">
                    <!-- Filters -->
                    <div class="row mb-4">
                        <div class="col-md-3">
                            <x-warehouse.select2-dropdown
                                name="status"
                                id="status_filter"
                                label="Status"
                                placeholder="All Status"
                                :options="[
                                    ['value' => '', 'text' => 'All Status'],
                                    ['value' => 'draft', 'text' => 'Draft'],
                                    ['value' => 'dispatched', 'text' => 'Dispatched'],
                                    ['value' => 'in_transit', 'text' => 'In Transit'],
                                    ['value' => 'delivered', 'text' => 'Delivered'],
                                    ['value' => 'completed', 'text' => 'Completed'],
                                    ['value' => 'cancelled', 'text' => 'Cancelled']
                                ]"
                                :selected="request('status')"
                                allowClear="true"
                            />
                        </div>
                        <div class="col-md-3">
                            <label for="issue_date_from">Issue Date From</label>
                            <input type="date" name="issue_date_from" id="issue_date_from" class="form-control" value="{{ request('issue_date_from') }}">
                        </div>
                        <div class="col-md-3">
                            <label for="issue_date_to">Issue Date To</label>
                            <input type="date" name="issue_date_to" id="issue_date_to" class="form-control" value="{{ request('issue_date_to') }}">
                        </div>
                        <div class="col-md-3 d-flex align-items-end">
                            <button type="button" onclick="applyFilters()" class="btn btn-info mr-2">
                                <i class="material-icons mr-1">filter_list</i>
                                Apply Filters
                            </button>
                            <a href="{{ route('warehouse.transfer-issues.index') }}" class="btn btn-outline-secondary">
                                <i class="material-icons mr-1">clear</i>
                                Clear
                            </a>
                        </div>
                    </div>

                    <!-- All Pending Transfer Requests (Debug Section) -->
                    @if($allPendingRequests->count() > 0)
                    <div class="mb-4">
                        <div class="card border-left-info">
                            <div class="card-header">
                                <h5 class="card-title mb-0">
                                    <i class="material-icons mr-2">list_alt</i>
                                    All Transfer Requests in Workflow
                                    <span class="badge badge-info ml-2">{{ $allPendingRequests->count() }}</span>
                                </h5>
                                <small class="text-muted">Transfer requests that need approval or are ready for issue creation</small>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-sm table-hover">
                                        <thead>
                                            <tr>
                                                <th>Request #</th>
                                                <th>Created Date</th>
                                                <th>From Project</th>
                                                <th>To Project</th>
                                                <th>Status</th>
                                                <th>Priority</th>
                                                <th>Items</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            @foreach($allPendingRequests as $request)
                                            <tr>
                                                <td>
                                                    <strong class="text-primary">{{ $request->transfer_request_number }}</strong>
                                                </td>
                                                <td>{{ $request->created_at->format('M d, Y') }}</td>
                                                <td>
                                                    <span class="text-primary">
                                                        {{ $request->transferrerProject->project_name }}
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="text-success">
                                                        {{ $request->receiverProject->project_name }}
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="badge
                                                        @if($request->status == 'approved') badge-success
                                                        @elseif($request->status == 'partially_approved') badge-info
                                                        @elseif(str_contains($request->status, 'pending')) badge-warning
                                                        @else badge-secondary
                                                        @endif">
                                                        @if($request->status == 'approved')
                                                            Transferred
                                                        @elseif($request->status == 'partially_approved')
                                                            Partially Transferred
                                                        @else
                                                            {{ ucwords(str_replace('_', ' ', $request->status)) }}
                                                        @endif
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="badge badge-{{ $request->priority_color }}">
                                                        {{ ucfirst($request->priority) }}
                                                    </span>
                                                </td>
                                                <td>
                                                    <span class="badge badge-info">{{ $request->items->count() }} items</span>
                                                </td>
                                                <td>
                                                    @if($request->status == 'approved')
                                                        <a href="{{ route('warehouse.transfer-requests.show', $request) }}"
                                                           class="btn btn-sm btn-outline-secondary mr-1"
                                                           title="View Details">
                                                            <i class="material-icons" style="font-size: 16px;">visibility</i>
                                                        </a>
                                                        <button class="btn btn-sm btn-outline-info mr-1"
                                                                title="Print Transfer Note"
                                                                onclick="printTransferRequest({{ $request->id }})">
                                                            <i class="material-icons" style="font-size: 16px;">print</i>
                                                        </button>
                                                    @else
                                                        <a href="{{ route('warehouse.transfer-requests.show', $request) }}"
                                                           class="btn btn-sm btn-outline-secondary mr-1"
                                                           title="View Details">
                                                            <i class="material-icons" style="font-size: 16px;">visibility</i>
                                                        </a>
                                                        @if(in_array($request->status, ['pending_transferrer_approval', 'partially_approved']))
                                                            <a href="{{ route('warehouse.transfer-requests.approve', $request) }}"
                                                               class="btn btn-sm btn-success mr-1"
                                                               title="{{ $request->status === 'partially_approved' ? 'Complete Transfer' : 'Dispatch Request' }}">
                                                                <i class="material-icons" style="font-size: 16px;">local_shipping</i>
                                                            </a>
                                                        @endif
                                                        <button class="btn btn-sm btn-outline-info mr-1"
                                                                title="Print Transfer Note"
                                                                onclick="printTransferRequest({{ $request->id }})">
                                                            <i class="material-icons" style="font-size: 16px;">print</i>
                                                        </button>
                                                    @endif
                                                </td>
                                            </tr>
                                            @endforeach
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    @endif

                </div>
            </div>
        </div>
    </div>
</div>

<!-- Dispatch Modal -->
<div class="modal fade" id="dispatchModal" tabindex="-1" role="dialog" aria-labelledby="dispatchModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="dispatchModalLabel">Dispatch Transfer</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form id="dispatchForm" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="form-group">
                        <label for="dispatch_vehicle">Vehicle Number</label>
                        <input type="text" name="dispatch_vehicle" id="dispatch_vehicle" class="form-control">
                    </div>
                    <div class="form-group">
                        <label for="driver_name">Driver Name</label>
                        <input type="text" name="driver_name" id="driver_name" class="form-control">
                    </div>
                    <div class="form-group">
                        <label for="driver_contact">Driver Contact</label>
                        <input type="text" name="driver_contact" id="driver_contact" class="form-control">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Dispatch Transfer</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Receive Modal -->
<div class="modal fade" id="receiveModal" tabindex="-1" role="dialog" aria-labelledby="receiveModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="receiveModalLabel">Receive Transfer</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form id="receiveForm" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="form-group">
                        <label for="receipt_notes">Receipt Notes</label>
                        <textarea name="receipt_notes" id="receipt_notes" class="form-control" rows="3"></textarea>
                    </div>
                    <div id="receive-items-container">
                        <!-- Items will be loaded here -->
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">Complete Receipt</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function applyFilters() {
    const params = new URLSearchParams();

    const status = $('#status_filter').val();
    const dateFrom = document.getElementById('issue_date_from').value;
    const dateTo = document.getElementById('issue_date_to').value;

    if (status) params.append('status', status);
    if (dateFrom) params.append('issue_date_from', dateFrom);
    if (dateTo) params.append('issue_date_to', dateTo);

    window.location.href = '{{ route("warehouse.transfer-issues.index") }}?' + params.toString();
}


function dispatchTransfer(issueId) {
    document.getElementById('dispatchForm').action = `/warehouse/transfer-issues/${issueId}/dispatch`;
    $('#dispatchModal').modal('show');
}

function markInTransit(issueId) {
    if (confirm('Mark this transfer as in transit?')) {
        fetch(`/warehouse/transfer-issues/${issueId}/mark-in-transit`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Error: ' + (data.message || 'Unknown error'));
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error marking transfer as in transit');
        });
    }
}

function receiveTransfer(issueId) {
    // Load transfer items for receipt
    fetch(`/warehouse/transfer-issues/${issueId}`)
        .then(response => response.text())
        .then(html => {
            // This would need to be implemented to load items data
            document.getElementById('receiveForm').action = `/warehouse/transfer-issues/${issueId}/receive`;
            $('#receiveModal').modal('show');
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading transfer details');
        });
}

// Print function for transfer requests
function printTransferRequest(id) {
    console.log('printTransferRequest called with id:', id);

    // Open print view in new window - using Laravel route helper
    const printUrl = '{{ route("warehouse.transfer-requests.print", ":id") }}'.replace(':id', id);
    console.log('Opening print URL:', printUrl);
    const printWindow = window.open(printUrl, '_blank', 'width=800,height=600,scrollbars=yes,resizable=yes');

    if (printWindow) {
        printWindow.focus();

        // Wait for content to load then print
        printWindow.addEventListener('load', function() {
            setTimeout(() => {
                printWindow.print();
            }, 1000);
        });
    } else {
        // Fallback: navigate to print page if popup blocked
        window.open(printUrl, '_blank');
    }
}
</script>

@endsection