@extends('layouts.admin-simple')

@section('title', 'Supplier Delivery Report')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.index') }}">Reports</a></li>
    <li class="breadcrumb-item active">Supplier Delivery</li>
@endsection

@section('page-title')
    <div class="page-title-left">
        <h3 class="mb-1">Supplier Delivery Report</h3>
        <p class="text-muted mb-0" style="font-size: 13px;">Supplier deliveries and purchase tracking</p>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4 warehouse-stats-row">
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Deliveries"
                :value="$incomingItems->total()"
                icon="local_shipping"
                color="primary"
                subtitle="All incoming deliveries"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Quantity"
                :value="number_format($incomingItems->getCollection()->sum('quantity'), 2)"
                icon="inventory_2"
                color="info"
                subtitle="Total quantity delivered"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Value"
                :value="'AED ' . number_format($incomingItems->getCollection()->sum('total_price'), 2)"
                icon="attach_money"
                color="success"
                subtitle="Total value received"
                :showDirham="true"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Pending Deliveries"
                :value="collect($incomingItems->getCollection())->where('status', 'pending')->count()"
                icon="schedule"
                color="warning"
                subtitle="Awaiting completion"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4 warehouse-filters-panel">
        <div class="card-header">
            <h5 class="mb-0">
                <i class="material-icons align-middle mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <form id="filterForm" class="warehouse-filter-form" method="GET" action="{{ route('warehouse.reports.incoming') }}">
                <div class="row">
                    <div class="col-md-2 mb-3">
                        <label class="form-label">Search</label>
                        <input type="text" name="search" value="{{ request('search') }}" class="form-control" 
                               placeholder="PO number, item code">
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Supplier</label>
                        <select name="supplier_id" id="filter-supplier" class="form-control" data-placeholder="All Suppliers">
                            <option value="">All Suppliers</option>
                            @foreach($suppliers as $supplier)
                                <option value="{{ $supplier->id }}" {{ request('supplier_id') == $supplier->id ? 'selected' : '' }}>
                                    {{ $supplier->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Category</label>
                        <select name="category_id" id="filter-category" class="form-control" data-placeholder="All Categories">
                            <option value="">All Categories</option>
                            @foreach($categories as $category)
                                <option value="{{ $category->id }}" {{ request('category_id') == $category->id ? 'selected' : '' }}>
                                    {{ $category->category_name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Division</label>
                        <select name="division_id" id="filter-division" class="form-control" data-placeholder="All Divisions">
                            <option value="">All Divisions</option>
                            @foreach(\App\Models\Warehouse\ProjectDivision::where('status', 'active')->orderBy('division_name')->get() as $division)
                                <option value="{{ $division->id }}" {{ request('division_id') == $division->id ? 'selected' : '' }}>
                                    {{ $division->division_name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Date From</label>
                        <input type="date" name="date_from" value="{{ request('date_from') }}" class="form-control warehouse-datepicker">
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Date To</label>
                        <input type="date" name="date_to" value="{{ request('date_to') }}" class="form-control warehouse-datepicker">
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary btn-block">
                            <i class="material-icons" style="font-size: 18px; vertical-align: middle;">search</i>
                            Apply
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Export Buttons -->
    <div class="row warehouse-export-buttons">
        <div class="col-12 text-right">
            <button class="btn btn-success warehouse-export-btn" data-export="excel">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">download</i>
                Export Excel
            </button>
            <button class="btn btn-info warehouse-export-btn" data-export="pdf">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">picture_as_pdf</i>
                Export PDF
            </button>
            <button class="btn btn-secondary warehouse-export-btn" data-action="print">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">print</i>
                Print
            </button>
        </div>
    </div>

    <!-- Supplier Delivery Report Table -->
    <div class="card">
        <div class="card-header warehouse-report-header">
            <h5 class="mb-0 warehouse-report-title">
                <i class="material-icons align-middle mr-2">local_shipping</i>
                Supplier Delivery Report ({{ $incomingItems->total() }} total deliveries)
            </h5>
        </div>
        <div class="card-body p-0">
            <div class="warehouse-table-container">
                <!-- Loading Overlay -->
                <div class="warehouse-loading-overlay" id="tableLoadingOverlay">
                    <div class="warehouse-loading-content">
                        <div class="text-center">
                            <div class="spinner-border text-primary" role="status">
                                <span class="sr-only">Loading...</span>
                            </div>
                            <div class="mt-2">Loading delivery data...</div>
                        </div>
                    </div>
                </div>

                <!-- Table -->
                <div class="warehouse-table-wrapper">
                    <table class="table table-hover mb-0 warehouse-table">
                        <thead class="thead-light">
                            <tr>
                                <th style="min-width: 120px;">PO Number</th>
                                <th style="min-width: 100px;">Date</th>
                                <th style="min-width: 150px;">Supplier</th>
                                <th style="min-width: 120px;">Division</th>
                                <th style="min-width: 120px;">Item Code</th>
                                <th style="min-width: 180px;">Description</th>
                                <th style="min-width: 100px;">Category</th>
                                <th style="min-width: 80px;" class="text-right">Ordered</th>
                                <th style="min-width: 80px;" class="text-right">Delivered</th>
                                <th style="min-width: 100px;" class="text-right">Unit Price</th>
                                <th style="min-width: 100px;" class="text-right">Total Price</th>
                                <th style="min-width: 100px;" class="text-center">Status</th>
                                <th style="min-width: 80px;" class="text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($incomingItems as $item)
                                <tr>
                                    <td class="warehouse-cell-nowrap">{{ $item->lpo_number ?? '-' }}</td>
                                    <td class="warehouse-cell-date">{{ $item->operation_date ? \Carbon\Carbon::parse($item->operation_date)->format('d M Y') : '-' }}</td>
                                    <td>{{ Str::limit($item->supplier_name ?? '-', 20) }}</td>
                                    <td>
                                        @if($item->division_name)
                                            <span class="badge warehouse-badge warehouse-status-info">
                                                {{ Str::limit($item->division_name, 15) }}
                                            </span>
                                        @else
                                            <span class="text-muted">-</span>
                                        @endif
                                    </td>
                                    <td class="warehouse-cell-nowrap">{{ $item->item_code }}</td>
                                    <td>{{ Str::limit($item->item_description, 30) }}</td>
                                    <td>
                                        <span class="badge warehouse-badge warehouse-status-secondary">
                                            No Category
                                        </span>
                                    </td>
                                    <td class="text-right warehouse-cell-number">
                                        @if($item->quantity_ordered)
                                            {{ number_format($item->quantity_ordered, 2) }}
                                        @else
                                            -
                                        @endif
                                    </td>
                                    <td class="text-right warehouse-cell-number">
                                        <strong>{{ number_format($item->quantity, 2) }}</strong>
                                    </td>
                                    <td class="text-right warehouse-cell-currency">
                                        AED {{ number_format($item->unit_price, 2) }}
                                    </td>
                                    <td class="text-right warehouse-cell-currency">
                                        <strong>AED {{ number_format($item->total_price, 2) }}</strong>
                                    </td>
                                    <td class="warehouse-cell-status">
                                        <span class="badge warehouse-badge warehouse-status-{{
                                            $item->status === 'completed' ? 'success' :
                                            ($item->status === 'partial' ? 'warning' : 
                                            ($item->status === 'pending' ? 'info' : 'danger'))
                                        }}">
                                            {{ ucfirst($item->status) }}
                                        </span>
                                    </td>
                                    <td class="text-center">
                                        @if($item->operation_id)
                                            <a href="{{ route('warehouse.incoming-operations.show', $item->operation_id) }}"
                                               class="btn btn-sm btn-outline-primary warehouse-action-btn" title="View Operation">
                                                <i class="material-icons">visibility</i>
                                            </a>
                                        @elseif($item->transaction_id ?? null)
                                            <a href="{{ route('warehouse.direct-deliveries.show', $item->transaction_id) }}"
                                               class="btn btn-sm btn-outline-primary warehouse-action-btn" title="View Transaction">
                                                <i class="material-icons">visibility</i>
                                            </a>
                                        @else
                                            <span class="text-muted">-</span>
                                        @endif
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="13" class="warehouse-empty-state">
                                        <i class="material-icons">local_shipping</i>
                                        <h5>No Delivery Records Found</h5>
                                        <p class="text-muted">No deliveries match your current filters.</p>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        @if($incomingItems->hasPages())
            <div class="card-footer">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        Showing {{ $incomingItems->firstItem() }} to {{ $incomingItems->lastItem() }} 
                        of {{ $incomingItems->total() }} deliveries
                    </div>
                    <div>
                        <nav class="warehouse-pagination">
                            {{ $incomingItems->links('pagination::bootstrap-4') }}
                        </nav>
                    </div>
                </div>
            </div>
        @endif
    </div>
@endsection

@push('styles')
    <link href="{{ asset('admin-assets/css/warehouse-reports.css') }}" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.min.css" rel="stylesheet" />
    <style>
        .select2-container--bootstrap-5 .select2-selection {
            min-height: 38px;
            border: 1px solid #ced4da;
        }
        .select2-container--bootstrap-5 .select2-selection--single .select2-selection__rendered {
            line-height: 36px;
        }
        .select2-container--bootstrap-5 .select2-selection--single .select2-selection__arrow {
            height: 36px;
        }
        .select2-container {
            width: 100% !important;
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script src="{{ asset('admin-assets/js/warehouse-reports.js') }}"></script>
    <script>
        $(document).ready(function() {
            // Initialize Select2 for filter dropdowns
            $('#filter-supplier, #filter-category, #filter-division').select2({
                theme: 'bootstrap-5',
                allowClear: true,
                placeholder: function() {
                    return $(this).data('placeholder') || 'Select...';
                }
            });
        });
    </script>
@endpush