<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('inventory', function (Blueprint $table) {
            // Supplier tracking fields
            $table->foreignId('supplier_id')->nullable()->after('division_id')->constrained('suppliers')->onDelete('set null');
            $table->string('supplier_batch_number', 100)->nullable()->after('batch_number');
            $table->string('purchase_order_number', 50)->nullable()->after('supplier_batch_number');

            // Delivery tracking fields
            $table->string('delivery_status', 50)->default('pending')->after('status');
            $table->enum('delivery_type', ['direct_delivery', 'warehouse_transfer', 'site_return', 'purchase_order'])->nullable()->after('delivery_status');
            $table->string('delivery_reference', 100)->nullable()->after('delivery_type');
            $table->timestamp('delivered_at')->nullable()->after('delivery_reference');
            $table->timestamp('dispatched_at')->nullable()->after('delivered_at');

            // Additional cost tracking
            $table->decimal('purchase_price', 10, 2)->nullable()->after('unit_price');
            $table->decimal('vat_amount', 10, 2)->nullable()->after('purchase_price');
            $table->string('currency', 3)->default('AED')->after('vat_amount');

            // Quality and compliance tracking
            $table->string('quality_status', 50)->nullable()->after('status');
            $table->text('quality_notes')->nullable()->after('quality_status');
            $table->boolean('compliance_verified')->default(false)->after('quality_notes');

            // Add indexes for better query performance
            $table->index('supplier_id');
            $table->index('delivery_status');
            $table->index('delivery_reference');
            $table->index('purchase_order_number');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('inventory', function (Blueprint $table) {
            // Drop indexes first
            $table->dropIndex(['supplier_id']);
            $table->dropIndex(['delivery_status']);
            $table->dropIndex(['delivery_reference']);
            $table->dropIndex(['purchase_order_number']);

            // Drop foreign key constraint
            $table->dropForeign(['supplier_id']);

            // Drop columns
            $table->dropColumn([
                'supplier_id',
                'supplier_batch_number',
                'purchase_order_number',
                'delivery_status',
                'delivery_type',
                'delivery_reference',
                'delivered_at',
                'dispatched_at',
                'purchase_price',
                'vat_amount',
                'currency',
                'quality_status',
                'quality_notes',
                'compliance_verified'
            ]);
        });
    }
};