<?php

namespace App\Services;

use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectDivision;
use App\Models\SyncHistory;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Exception;

class ExternalProjectImportService
{
    protected $externalConnection = 'external_projects';

    /**
     * Import projects from external database
     *
     * @param array $options Import options (table_name, mapping, etc.)
     * @return array Import results
     */
    public function importProjects($options = [])
    {
        $startTime = now();

        // Create sync history record
        $syncHistory = SyncHistory::create([
            'sync_type' => 'projects',
            'external_table' => $options['table_name'] ?? 'project',
            'sync_options' => $options,
            'status' => 'running',
            'started_at' => $startTime,
            'triggered_by' => Auth::user()->name ?? 'System'
        ]);

        DB::beginTransaction();

        try {
            $externalTableName = $options['table_name'] ?? 'project';
            $mapping = $options['mapping'] ?? $this->getDefaultMapping();
            $batchSize = $options['batch_size'] ?? 100;
            $skipExisting = $options['skip_existing'] ?? true;
            $dryRun = $options['dry_run'] ?? false;

            $updateExisting = $options['update_existing'] ?? true; // NEW: Option to update existing records

            $results = [
                'total_processed' => 0,
                'imported' => 0,
                'updated' => 0, // NEW: Track updated records
                'skipped' => 0,
                'errors' => 0,
                'validation_errors' => 0,
                'error_details' => [],
                'validation_details' => [],
                'division_imports' => [],
                'division_sync' => [], // NEW: Track division sync results
                'sync_history_id' => $syncHistory->id
            ];

            // Test external database connection
            $this->testExternalConnection();

            // Pre-validate external data structure
            $this->validateExternalDataStructure($externalTableName);

            // STEP 1: Sync ALL divisions FIRST before syncing projects
            Log::info('Starting division sync (Step 1 of 2)...');
            $divisionResults = $this->syncAllDivisions();
            $results['division_sync'] = $divisionResults;
            $results['division_imports'] = [
                'total' => $divisionResults['total'],
                'created' => $divisionResults['created'],
                'updated' => $divisionResults['updated'],
                'errors' => $divisionResults['errors']
            ];
            Log::info('Division sync completed', $divisionResults);

            // STEP 2: Now sync projects
            Log::info('Starting project sync (Step 2 of 2)...');

            // Get external projects with validation
            $externalProjects = $this->getExternalProjects($externalTableName, $mapping);
            $results['total_processed'] = count($externalProjects);

            if (empty($externalProjects)) {
                Log::info('No external projects found to import');
                $this->updateSyncHistory($syncHistory, $results, 'completed');
                DB::commit(); // Commit division changes even if no projects
                return $results;
            }

            // DON'T filter out synced projects if we want to update them
            $projectsToProcess = $externalProjects;
            if ($skipExisting && !$updateExisting) {
                $projectsToProcess = $this->filterAlreadySyncedProjects($externalProjects);
                Log::info("After filtering already synced projects: " . count($projectsToProcess) . " remain");
            } else {
                Log::info("Processing all " . count($projectsToProcess) . " projects (update_existing: " . ($updateExisting ? 'true' : 'false') . ")");
            }

            // Pre-import data validation
            $validationResults = $this->validateProjectData($projectsToProcess);
            $results['validation_errors'] = $validationResults['errors'];
            $results['validation_details'] = $validationResults['details'];

            if ($validationResults['errors'] > 0) {
                Log::warning("Data validation found {$validationResults['errors']} issues");
            }

            // If dry run, return validation results without importing
            if ($dryRun) {
                $results['dry_run'] = true;
                $this->updateSyncHistory($syncHistory, $results, 'completed');
                DB::rollBack();
                return $results;
            }

            // Process projects in batches with improved error handling
            $batches = array_chunk($projectsToProcess, $batchSize);

            foreach ($batches as $batchIndex => $batch) {
                try {
                    $batchResults = $this->processBatchWithUpdate($batch, $skipExisting, $updateExisting, $batchIndex);
                    $results['imported'] += $batchResults['imported'];
                    $results['updated'] += $batchResults['updated'] ?? 0;
                    $results['skipped'] += $batchResults['skipped'];
                    $results['errors'] += $batchResults['errors'];
                    $results['error_details'] = array_merge($results['error_details'], $batchResults['error_details']);
                } catch (Exception $e) {
                    Log::error("Batch {$batchIndex} failed: " . $e->getMessage());
                    $results['errors']++;
                    $results['error_details'][] = "Batch {$batchIndex} failed: " . $e->getMessage();
                    // Continue with next batch instead of failing completely
                }
            }

            // Final integrity check
            $integrityCheck = $this->performIntegrityCheck();
            if (!$integrityCheck['passed']) {
                Log::warning('Integrity check failed', $integrityCheck);
                $results['integrity_warnings'] = $integrityCheck['warnings'];
            }

            // Update sync history with final results
            $this->updateSyncHistory($syncHistory, $results, 'completed');

            DB::commit();
            Log::info('Project import completed successfully', $results);
            return $results;

        } catch (Exception $e) {
            $this->updateSyncHistory($syncHistory, $results ?? [], 'failed', $e->getMessage());
            DB::rollBack();
            Log::error('Project import failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get external projects with specified mapping
     */
    protected function getExternalProjects($tableName, $mapping)
    {
        // Get all available columns first to avoid column not found errors
        try {
            return DB::connection($this->externalConnection)
                ->table($tableName)
                ->select('*')
                ->get()
                ->toArray();
        } catch (Exception $e) {
            Log::error("Failed to fetch external projects: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Process a batch of projects
     */
    protected function processBatch($projects, $skipExisting = true, $batchIndex = 0)
    {
        $results = [
            'imported' => 0,
            'skipped' => 0,
            'errors' => 0,
            'error_details' => [],
            'division_imports' => []
        ];

        foreach ($projects as $projectIndex => $externalProject) {
            try {
                // Validate individual project data
                $validation = $this->validateSingleProject($externalProject);
                if (!$validation['valid']) {
                    $results['skipped']++;  // Count as skipped, not error
                    $projectInfo = "Project ID: " . $this->cleanUtf8($externalProject->project_id ?? 'N/A') .
                                  ", No: " . $this->cleanUtf8($externalProject->project_no ?? 'N/A') .
                                  ", Name: " . substr($this->cleanUtf8($externalProject->project_name ?? 'N/A'), 0, 30);
                    $error = $projectInfo . " - Validation: " . implode(', ', $validation['errors']);
                    $results['error_details'][] = $error;
                    Log::warning("Skipping invalid project: {$error}");
                    continue;
                }

                $projectData = $this->mapProjectData($externalProject);

                // Enhanced duplicate detection
                $existingProject = $this->findExistingProject($projectData);
                if ($existingProject && $skipExisting) {
                    $results['skipped']++;
                    $skipInfo = "Project No: " . $this->cleanUtf8($projectData['project_number'] ?? 'N/A') .
                               ", Name: " . substr($this->cleanUtf8($projectData['project_name'] ?? ''), 0, 50) .
                               " - Already exists (ID: {$existingProject->id})";
                    $results['error_details'][] = $skipInfo;
                    Log::debug("Skipped existing project: {$skipInfo}");
                    continue;
                }

                // Create project with additional validation
                $project = $this->createProjectSafely($projectData, $externalProject);

                if ($project) {
                    $results['imported']++;
                    Log::info("Imported project: {$project->project_name} ({$project->project_number}) in batch {$batchIndex}");
                } else {
                    $results['errors']++;
                    $results['error_details'][] = "Failed to create project for unknown reason";
                }

            } catch (Exception $e) {
                $results['skipped']++;  // Count creation failures as skipped
                $projectInfo = "Project ID: " . $this->cleanUtf8($externalProject->project_id ?? 'N/A') .
                              ", No: " . $this->cleanUtf8($externalProject->project_no ?? 'N/A') .
                              ", Name: " . substr($this->cleanUtf8($externalProject->project_name ?? 'N/A'), 0, 30);
                $error = $projectInfo . " - Error: " . $this->cleanUtf8($e->getMessage());
                $results['error_details'][] = $error;
                Log::error("Failed to import project: {$error}");
            }
        }

        return $results;
    }

    /**
     * Process a batch of projects with update capability
     * This method creates new projects and updates existing ones if data has changed
     */
    protected function processBatchWithUpdate($projects, $skipExisting = true, $updateExisting = true, $batchIndex = 0)
    {
        $results = [
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => 0,
            'error_details' => []
        ];

        foreach ($projects as $projectIndex => $externalProject) {
            try {
                // Validate individual project data
                $validation = $this->validateSingleProject($externalProject);
                if (!$validation['valid']) {
                    $results['skipped']++;
                    $projectInfo = "Project ID: " . $this->cleanUtf8($externalProject->project_id ?? 'N/A') .
                                  ", No: " . $this->cleanUtf8($externalProject->project_no ?? 'N/A') .
                                  ", Name: " . substr($this->cleanUtf8($externalProject->project_name ?? 'N/A'), 0, 30);
                    $error = $projectInfo . " - Validation: " . implode(', ', $validation['errors']);
                    $results['error_details'][] = $error;
                    Log::warning("Skipping invalid project: {$error}");
                    continue;
                }

                $projectData = $this->mapProjectData($externalProject);

                // Check if project already exists
                $existingProject = $this->findExistingProject($projectData);

                if ($existingProject) {
                    if ($updateExisting) {
                        // Update existing project if data has changed
                        $updateResult = $this->updateExistingProject($existingProject, $projectData, $externalProject);
                        if ($updateResult['updated']) {
                            $results['updated']++;
                            Log::info("Updated project: {$existingProject->project_name} (ID: {$existingProject->id})", $updateResult['changes']);
                        } else {
                            $results['skipped']++;
                            Log::debug("Project unchanged: {$existingProject->project_name} (ID: {$existingProject->id})");
                        }
                    } else if ($skipExisting) {
                        $results['skipped']++;
                        Log::debug("Skipped existing project: {$existingProject->project_name}");
                    }
                    continue;
                }

                // Create new project
                $project = $this->createProjectSafely($projectData, $externalProject);

                if ($project) {
                    $results['imported']++;
                    Log::info("Imported project: {$project->project_name} ({$project->project_number}) in batch {$batchIndex}");
                } else {
                    $results['errors']++;
                    $results['error_details'][] = "Failed to create project for unknown reason";
                }

            } catch (Exception $e) {
                $results['skipped']++;
                $projectInfo = "Project ID: " . $this->cleanUtf8($externalProject->project_id ?? 'N/A') .
                              ", No: " . $this->cleanUtf8($externalProject->project_no ?? 'N/A') .
                              ", Name: " . substr($this->cleanUtf8($externalProject->project_name ?? 'N/A'), 0, 30);
                $error = $projectInfo . " - Error: " . $this->cleanUtf8($e->getMessage());
                $results['error_details'][] = $error;
                Log::error("Failed to process project: {$error}");
            }
        }

        return $results;
    }

    /**
     * Map external project data to local project structure
     */
    protected function mapProjectData($externalProject)
    {
        $projectData = [];

        // Map external project ID - CRITICAL for preventing duplicates
        $projectData['external_project_id'] = $externalProject->project_id ?? null;

        // Map ERP ID to project_id field
        $projectData['project_id'] = $externalProject->erp_id ?? null;

        // Map project name - clean UTF-8
        $projectData['project_name'] = $this->cleanUtf8($externalProject->project_name ?? '');

        // Handle division mapping using correct field name
        if (!empty($externalProject->division_id)) {
            $projectData['project_division_id'] = $this->mapProjectDivision($externalProject->division_id);
        } else {
            // Assign "Unassigned" division for projects without a division in source
            $projectData['project_division_id'] = $this->getUnassignedDivisionId();
        }

        // Use project number from external if available, otherwise generate
        if (!empty($externalProject->project_no)) {
            $projectData['project_number'] = $this->cleanUtf8($externalProject->project_no);
        } else {
            $projectData['project_number'] = Project::generateProjectNumber($projectData['project_division_id'] ?? null);
        }

        // Add sync timestamp
        $projectData['last_synced_at'] = now();

        return $projectData;
    }

    /**
     * Enhanced duplicate detection - PRIORITIZES external project ID to prevent re-imports
     */
    protected function findExistingProject($projectData)
    {
        // PRIMARY CHECK: by external project ID - this is the most reliable way to prevent duplicates
        if (!empty($projectData['external_project_id'])) {
            $project = Project::where('external_project_id', $projectData['external_project_id'])->first();
            if ($project) {
                Log::debug("Found existing project by external ID: {$projectData['external_project_id']} -> {$project->id}");
                return $project;
            }
        }

        // STRICT CHECK: Only consider duplicate if BOTH project number AND name match
        if (!empty($projectData['project_number']) && !empty($projectData['project_name'])) {
            $project = Project::where('project_number', $projectData['project_number'])
                             ->where('project_name', $projectData['project_name'])
                             ->first();
            if ($project) {
                Log::debug("Found existing project by BOTH number AND name: {$projectData['project_number']} / {$projectData['project_name']} -> {$project->id}");
                return $project;
            }
        }

        // If only project number matches but name is different -> NOT duplicate
        // If only project name matches but number is different -> NOT duplicate
        // Both number AND name must match to be considered a duplicate

        return null;
    }

    /**
     * Filter out projects that are already synced by external project ID
     */
    protected function filterAlreadySyncedProjects($externalProjects)
    {
        $externalIds = collect($externalProjects)->pluck('project_id')->filter()->toArray();

        if (empty($externalIds)) {
            Log::warning('No external project IDs found - cannot filter synced projects effectively');
            return $externalProjects;
        }

        // Get already synced project IDs
        $syncedIds = Project::whereIn('external_project_id', $externalIds)->pluck('external_project_id')->toArray();

        Log::info("Found " . count($syncedIds) . " already synced projects out of " . count($externalIds));

        // Filter out already synced projects
        return array_filter($externalProjects, function($project) use ($syncedIds) {
            return !in_array($project->project_id, $syncedIds);
        });
    }

    /**
     * Legacy method for backward compatibility
     */
    protected function projectExists($projectData)
    {
        return $this->findExistingProject($projectData) !== null;
    }

    /**
     * Get the ID of the "Unassigned" division
     * Creates it if it doesn't exist
     */
    protected function getUnassignedDivisionId()
    {
        static $unassignedDivisionId = null;

        if ($unassignedDivisionId === null) {
            $division = ProjectDivision::where('division_name', 'Unassigned')->first();

            if (!$division) {
                // Create the Unassigned division if it doesn't exist
                $division = ProjectDivision::create([
                    'division_name' => 'Unassigned',
                    'division_code' => 'UNA',
                    'description' => 'Default division for projects without an assigned division',
                    'status' => 'active',
                    'external_division_id' => null
                ]);
                Log::info("Created 'Unassigned' division with ID: {$division->id}");
            }

            $unassignedDivisionId = $division->id;
        }

        return $unassignedDivisionId;
    }

    /**
     * Map external project division to local division
     */
    protected function mapProjectDivision($externalDivisionId)
    {
        // First try to find existing division by external_division_id
        $division = ProjectDivision::where('external_division_id', $externalDivisionId)->first();

        if (!$division) {
            // If not found, try to import the division from external database
            $division = $this->importDivision($externalDivisionId);
        }

        return $division ? $division->id : null;
    }

    /**
     * Import division from external database
     */
    protected function importDivision($externalDivisionId)
    {
        try {
            $externalDivision = DB::connection($this->externalConnection)
                ->table('division')
                ->where('division_id', $externalDivisionId)
                ->first();

            if ($externalDivision) {
                $divisionName = $this->cleanUtf8($externalDivision->name);
                $divisionCode = $this->generateUniqueDivisionCode($divisionName);

                // Create new project division
                $division = ProjectDivision::create([
                    'division_name' => $divisionName,
                    'division_code' => $divisionCode,
                    'external_division_id' => $externalDivisionId,
                    'status' => 'active'
                ]);

                Log::info("Imported division: {$division->division_name} (Code: {$divisionCode}, ID: {$externalDivisionId})");
                return $division;
            }

            return null;
        } catch (Exception $e) {
            Log::error("Failed to import division {$externalDivisionId}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Generate a unique division code from the division name
     * Uses abbreviations from name parts to create more meaningful codes
     *
     * @param string $divisionName
     * @return string Unique division code
     */
    protected function generateUniqueDivisionCode($divisionName)
    {
        // First, try to create a more meaningful code from the name parts
        $nameParts = explode(' ', strtoupper(trim($divisionName)));

        if (count($nameParts) >= 2) {
            // Use first letters of each word (e.g., "CRES DUBAI" -> "CD", "BMC ABU DHABI" -> "BAD")
            $code = '';
            foreach ($nameParts as $part) {
                $code .= substr($part, 0, 1);
            }
            $code = substr($code, 0, 5); // Max 5 chars

            // Check if this code is unique
            if (!ProjectDivision::where('division_code', $code)->exists()) {
                return $code;
            }

            // Try first 2 letters of first word + first letter of second word
            if (count($nameParts) >= 2) {
                $code = substr($nameParts[0], 0, 2) . substr($nameParts[1], 0, 1);
                if (!ProjectDivision::where('division_code', $code)->exists()) {
                    return $code;
                }

                // Try first letter of first word + first 2 letters of second word
                $code = substr($nameParts[0], 0, 1) . substr($nameParts[1], 0, 2);
                if (!ProjectDivision::where('division_code', $code)->exists()) {
                    return $code;
                }
            }
        }

        // Fallback: First 3 chars + incrementing number
        $baseCode = strtoupper(substr($divisionName, 0, 3));
        $code = $baseCode;
        $counter = 1;

        while (ProjectDivision::where('division_code', $code)->exists() && $counter < 100) {
            $code = $baseCode . $counter;
            $counter++;
        }

        return $code;
    }

    /**
     * Sync ALL divisions from external database first
     * This ensures all divisions exist before syncing projects
     * Also updates existing divisions if data has changed
     *
     * @return array Results of division sync
     */
    public function syncAllDivisions()
    {
        $results = [
            'total' => 0,
            'created' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => 0,
            'error_details' => []
        ];

        try {
            // Get all divisions from external database
            $externalDivisions = DB::connection($this->externalConnection)
                ->table('division')
                ->get();

            $results['total'] = $externalDivisions->count();
            Log::info("Found {$results['total']} divisions in external database");

            foreach ($externalDivisions as $externalDivision) {
                try {
                    $externalId = $externalDivision->division_id;
                    $divisionName = $this->cleanUtf8($externalDivision->name ?? '');

                    if (empty($divisionName)) {
                        $results['skipped']++;
                        $results['error_details'][] = "Division ID {$externalId}: Empty name, skipped";
                        continue;
                    }

                    // Check if division already exists by external_division_id
                    $existingDivision = ProjectDivision::where('external_division_id', $externalId)->first();

                    if ($existingDivision) {
                        // Check if data has changed and needs update
                        $needsUpdate = false;
                        $changes = [];

                        if ($existingDivision->division_name !== $divisionName) {
                            $changes['division_name'] = $divisionName;
                            $needsUpdate = true;
                        }

                        // Don't update division_code for existing divisions (they already have unique codes)

                        if ($needsUpdate) {
                            $existingDivision->update($changes);
                            $results['updated']++;
                            Log::info("Updated division: {$divisionName} (External ID: {$externalId})", $changes);
                        } else {
                            $results['skipped']++;
                            Log::debug("Division unchanged: {$divisionName} (External ID: {$externalId})");
                        }
                    } else {
                        // Check if division exists by name (might have been created manually)
                        $existingByName = ProjectDivision::where('division_name', $divisionName)->first();

                        if ($existingByName) {
                            // Link existing division to external ID
                            $existingByName->update(['external_division_id' => $externalId]);
                            $results['updated']++;
                            Log::info("Linked existing division to external ID: {$divisionName} -> {$externalId}");
                        } else {
                            // Create new division with unique code
                            $divisionCode = $this->generateUniqueDivisionCode($divisionName);
                            $division = ProjectDivision::create([
                                'division_name' => $divisionName,
                                'division_code' => $divisionCode,
                                'external_division_id' => $externalId,
                                'status' => 'active'
                            ]);
                            $results['created']++;
                            Log::info("Created division: {$divisionName} (Code: {$divisionCode}, External ID: {$externalId})");
                        }
                    }
                } catch (Exception $e) {
                    $results['errors']++;
                    $errorMsg = "Division ID " . ($externalDivision->division_id ?? 'N/A') . ": " . $e->getMessage();
                    $results['error_details'][] = $errorMsg;
                    Log::error("Failed to sync division: " . $errorMsg);
                }
            }

            Log::info("Division sync completed", $results);
            return $results;

        } catch (Exception $e) {
            Log::error("Division sync failed: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update existing project if data has changed
     *
     * @param Project $existingProject
     * @param array $projectData
     * @param object $externalProject
     * @return array ['updated' => bool, 'changes' => array]
     */
    protected function updateExistingProject($existingProject, $projectData, $externalProject)
    {
        $changes = [];
        $needsUpdate = false;

        // Check project name
        if (!empty($projectData['project_name']) && $existingProject->project_name !== $projectData['project_name']) {
            $changes['project_name'] = $projectData['project_name'];
            $needsUpdate = true;
        }

        // Check project number
        if (!empty($projectData['project_number']) && $existingProject->project_number !== $projectData['project_number']) {
            // Only update if the new number doesn't conflict with another project
            $conflictingProject = Project::where('project_number', $projectData['project_number'])
                ->where('id', '!=', $existingProject->id)
                ->first();

            if (!$conflictingProject) {
                $changes['project_number'] = $projectData['project_number'];
                $needsUpdate = true;
            }
        }

        // Check division - also assign Unassigned division if project has no division
        if (!empty($projectData['project_division_id']) && $existingProject->project_division_id !== $projectData['project_division_id']) {
            $changes['project_division_id'] = $projectData['project_division_id'];
            $needsUpdate = true;
        } elseif (empty($existingProject->project_division_id)) {
            // Assign Unassigned division to existing projects without a division
            $changes['project_division_id'] = $this->getUnassignedDivisionId();
            $needsUpdate = true;
            Log::info("Assigning 'Unassigned' division to project: {$existingProject->project_name}");
        }

        // Check ERP ID (project_id field)
        if (!empty($projectData['project_id']) && $existingProject->project_id !== $projectData['project_id']) {
            $changes['project_id'] = $projectData['project_id'];
            $needsUpdate = true;
        }

        // Always update sync timestamp
        $changes['last_synced_at'] = now();

        if ($needsUpdate) {
            $existingProject->update($changes);
            Log::info("Updated existing project: {$existingProject->project_name}", [
                'project_id' => $existingProject->id,
                'changes' => $changes
            ]);
            return ['updated' => true, 'changes' => $changes];
        }

        // Just update sync timestamp even if no other changes
        $existingProject->update(['last_synced_at' => now()]);
        return ['updated' => false, 'changes' => []];
    }

    /**
     * Get default field mapping
     */
    protected function getDefaultMapping()
    {
        return [
            'id' => 'external_project_id',
            'project_name' => 'project_name',
            'division_id' => 'external_division_id',
        ];
    }

    /**
     * Test external database connection
     */
    protected function testExternalConnection()
    {
        try {
            DB::connection($this->externalConnection)->getPdo();
            Log::info('External database connection successful');
        } catch (Exception $e) {
            throw new Exception("Cannot connect to external database: " . $e->getMessage());
        }
    }

    /**
     * Get external database schema information
     */
    public function getExternalDatabaseInfo()
    {
        try {
            $this->testExternalConnection();

            $tables = DB::connection($this->externalConnection)
                ->select('SHOW TABLES');

            $info = [
                'connection_status' => 'connected',
                'tables' => array_map(function($table) {
                    return array_values((array)$table)[0];
                }, $tables)
            ];

            return $info;
        } catch (Exception $e) {
            return [
                'connection_status' => 'failed',
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Get table structure for a specific table
     */
    public function getTableStructure($tableName)
    {
        try {
            $columns = DB::connection($this->externalConnection)
                ->select("DESCRIBE {$tableName}");

            return $columns;
        } catch (Exception $e) {
            throw new Exception("Cannot get table structure: " . $e->getMessage());
        }
    }

    /**
     * Get sample data from external tables to understand structure
     */
    public function inspectExternalTables()
    {
        try {
            $result = [];

            // Check project table structure
            $projectColumns = $this->getTableStructure('project');
            $result['project_structure'] = $projectColumns;

            // Get sample project data
            $sampleProject = DB::connection($this->externalConnection)
                ->table('project')
                ->select('project_id', 'project_no', 'project_name', 'division_id')
                ->first();
            $result['sample_project'] = $sampleProject;

            // Check division table structure
            $divisionColumns = $this->getTableStructure('division');
            $result['division_structure'] = $divisionColumns;

            // Get sample division data
            $sampleDivision = DB::connection($this->externalConnection)
                ->table('division')
                ->first();
            $result['sample_division'] = $sampleDivision;

            return $result;
        } catch (Exception $e) {
            throw new Exception("Cannot inspect external tables: " . $e->getMessage());
        }
    }

    /**
     * Validate external data structure before import
     */
    protected function validateExternalDataStructure($tableName)
    {
        try {
            // Check if required tables exist
            $tables = DB::connection($this->externalConnection)
                ->select('SHOW TABLES');

            $availableTables = array_map(function($table) {
                return array_values((array)$table)[0];
            }, $tables);

            if (!in_array($tableName, $availableTables)) {
                throw new Exception("Table '{$tableName}' not found in external database");
            }

            if (!in_array('division', $availableTables)) {
                throw new Exception("Division table not found in external database");
            }

            // Check if required columns exist
            $columns = $this->getTableStructure($tableName);
            $columnNames = array_column($columns, 'Field');

            $requiredColumns = ['project_name', 'project_no', 'division_id'];
            foreach ($requiredColumns as $required) {
                if (!in_array($required, $columnNames)) {
                    Log::warning("Optional column '{$required}' not found in {$tableName}");
                }
            }

            Log::info("External data structure validation passed");
        } catch (Exception $e) {
            Log::error("External data structure validation failed: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Validate project data before import
     */
    protected function validateProjectData($projects)
    {
        $results = [
            'errors' => 0,
            'details' => []
        ];

        $projectNumbers = [];
        $projectNames = [];

        foreach ($projects as $index => $project) {
            $validation = $this->validateSingleProject($project);
            if (!$validation['valid']) {
                $results['errors']++;
                $results['details'][] = "Project {$index}: " . implode(', ', $validation['errors']);
            }

            // Check for duplicates within the import set
            if (!empty($project->project_no)) {
                if (in_array($project->project_no, $projectNumbers)) {
                    $results['errors']++;
                    $results['details'][] = "Duplicate project number in import set: {$project->project_no}";
                } else {
                    $projectNumbers[] = $project->project_no;
                }
            }

            if (!empty($project->project_name)) {
                if (in_array($project->project_name, $projectNames)) {
                    $results['errors']++;
                    $results['details'][] = "Duplicate project name in import set: {$project->project_name}";
                } else {
                    $projectNames[] = $project->project_name;
                }
            }
        }

        return $results;
    }

    /**
     * Validate single project data
     */
    protected function validateSingleProject($project)
    {
        $errors = [];

        // Check required fields
        if (empty($project->project_name) || trim($project->project_name) === '') {
            $errors[] = "Missing project name";
        }

        // Validate project name length
        if (!empty($project->project_name) && strlen($project->project_name) > 255) {
            $errors[] = "Project name too long (max 255 characters)";
        }

        // Validate project number format if present
        if (!empty($project->project_no)) {
            if (strlen($project->project_no) > 50) {
                $errors[] = "Project number too long (max 50 characters)";
            }
        }

        // Validate division ID if present
        if (!empty($project->division_id)) {
            if (!is_numeric($project->division_id)) {
                $errors[] = "Invalid division ID format";
            }
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }

    /**
     * Create project safely with additional validation
     */
    protected function createProjectSafely($projectData, $externalProject)
    {
        try {
            // Final validation before creation
            if (empty($projectData['project_name'])) {
                throw new Exception("Cannot create project without name");
            }

            // Check if project number exists with different name - this means multiple projects share same number
            if (!empty($projectData['project_number'])) {
                $existing = Project::where('project_number', $projectData['project_number'])
                                  ->where('project_name', '!=', $projectData['project_name'])
                                  ->first();
                if ($existing) {
                    // Different project with same number - generate unique project number
                    $originalNumber = $projectData['project_number'];
                    $counter = 1;
                    do {
                        $newNumber = $originalNumber . '-' . $counter;
                        $numberExists = Project::where('project_number', $newNumber)->exists();
                        $counter++;
                    } while ($numberExists && $counter < 100); // Safety limit

                    $projectData['project_number'] = $newNumber;
                    Log::warning("Project number conflict resolved: {$originalNumber} -> {$newNumber} for project: {$projectData['project_name']}");
                }
            }

            // Create the project
            $project = Project::create($projectData);

            // Log the creation with external reference
            Log::info("Created project safely", [
                'internal_id' => $project->id,
                'project_number' => $project->project_number,
                'external_project_id' => $externalProject->project_id ?? null
            ]);

            return $project;

        } catch (Exception $e) {
            Log::error("Failed to create project safely: " . $e->getMessage(), [
                'project_data' => $projectData,
                'external_data' => $externalProject
            ]);
            throw $e;
        }
    }

    /**
     * Perform integrity check after import
     */
    protected function performIntegrityCheck()
    {
        $warnings = [];
        $passed = true;

        try {
            // Check for projects without divisions
            $projectsWithoutDivisions = Project::whereNull('project_division_id')->count();
            if ($projectsWithoutDivisions > 0) {
                $warnings[] = "{$projectsWithoutDivisions} projects have no division assigned";
                $passed = false;
            }

            // Check for duplicate project numbers
            $duplicateNumbers = DB::table('projects')
                ->select('project_number')
                ->whereNotNull('project_number')
                ->groupBy('project_number')
                ->havingRaw('COUNT(*) > 1')
                ->get();

            if ($duplicateNumbers->count() > 0) {
                $warnings[] = "Found {$duplicateNumbers->count()} duplicate project numbers";
                $passed = false;
            }

            // Check for projects with empty names
            $emptyNames = Project::where('project_name', '')->orWhereNull('project_name')->count();
            if ($emptyNames > 0) {
                $warnings[] = "{$emptyNames} projects have empty names";
                $passed = false;
            }

            // Check division consistency
            $orphanedDivisions = ProjectDivision::whereDoesntHave('projects')->count();
            if ($orphanedDivisions > 0) {
                $warnings[] = "{$orphanedDivisions} divisions have no projects";
            }

        } catch (Exception $e) {
            $warnings[] = "Integrity check failed: " . $e->getMessage();
            $passed = false;
        }

        return [
            'passed' => $passed,
            'warnings' => $warnings
        ];
    }

    /**
     * Update sync history record with results
     */
    protected function updateSyncHistory(SyncHistory $syncHistory, array $results, string $status, ?string $errorMessage = null)
    {
        $updateData = [
            'total_processed' => $results['total_processed'] ?? 0,
            'imported' => $results['imported'] ?? 0,
            'updated' => $results['updated'] ?? 0, // Track updated records
            'skipped' => $results['skipped'] ?? 0,
            'errors' => $results['errors'] ?? 0,
            'validation_errors' => $results['validation_errors'] ?? 0,
            'error_details' => $this->cleanErrorDetailsArray($results['error_details'] ?? []),
            'validation_details' => $this->cleanErrorDetailsArray($results['validation_details'] ?? []),
            'division_sync' => $results['division_sync'] ?? [], // Track division sync results
            'status' => $status,
            'completed_at' => now(),
            'duration_seconds' => now()->diffInSeconds($syncHistory->started_at)
        ];

        if ($errorMessage) {
            $updateData['error_details'][] = $this->cleanUtf8($errorMessage);
        }

        $syncHistory->update($updateData);

        Log::info("Updated sync history {$syncHistory->id} with status: {$status}");
    }

    /**
     * Get statistics about external project IDs in current database
     */
    public function getExternalIdStatistics()
    {
        $total = Project::count();
        $withExternalIds = Project::whereNotNull('external_project_id')->count();
        $withoutExternalIds = $total - $withExternalIds;

        $recentSyncs = SyncHistory::recent(7)->count();
        $lastSync = SyncHistory::getLastSuccessfulSync('projects');

        return [
            'total_projects' => $total,
            'with_external_ids' => $withExternalIds,
            'without_external_ids' => $withoutExternalIds,
            'external_id_coverage' => $total > 0 ? round(($withExternalIds / $total) * 100, 2) : 0,
            'recent_syncs_7_days' => $recentSyncs,
            'last_successful_sync' => $lastSync ? $lastSync->completed_at->format('Y-m-d H:i:s') : null
        ];
    }

    /**
     * Backfill external project IDs for existing projects
     */
    public function backfillExternalIds($options = [])
    {
        $batchSize = $options['batch_size'] ?? 100;
        $results = [
            'processed' => 0,
            'updated' => 0,
            'not_found' => 0,
            'errors' => 0
        ];

        try {
            // Get projects without external IDs
            $projectsWithoutIds = Project::whereNull('external_project_id')
                ->whereNotNull('project_number')
                ->get();

            $results['processed'] = $projectsWithoutIds->count();

            if ($results['processed'] === 0) {
                Log::info('No projects found that need external ID backfill');
                return $results;
            }

            Log::info("Starting backfill for {$results['processed']} projects");

            foreach ($projectsWithoutIds->chunk($batchSize) as $chunk) {
                foreach ($chunk as $project) {
                    try {
                        // Try to find matching external project
                        $externalProject = DB::connection($this->externalConnection)
                            ->table('project')
                            ->where('project_no', $project->project_number)
                            ->first();

                        if ($externalProject) {
                            $project->update([
                                'external_project_id' => $externalProject->project_id,
                                'project_id' => $externalProject->erp_id ?? null,
                                'last_synced_at' => now()
                            ]);
                            $results['updated']++;
                            Log::debug("Backfilled external ID {$externalProject->project_id} and ERP ID {$externalProject->erp_id} for project {$project->id}");
                        } else {
                            $results['not_found']++;
                            Log::debug("No external project found for {$project->project_number}");
                        }
                    } catch (Exception $e) {
                        $results['errors']++;
                        Log::error("Error backfilling project {$project->id}: " . $e->getMessage());
                    }
                }
            }

            Log::info('External ID backfill completed', $results);
            return $results;

        } catch (Exception $e) {
            Log::error('External ID backfill failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Clean UTF-8 characters to prevent JSON encoding errors
     */
    protected function cleanUtf8($string)
    {
        if (!is_string($string)) {
            return $string;
        }

        // First, try to detect and convert encoding
        $encoding = mb_detect_encoding($string, ['UTF-8', 'ISO-8859-1', 'Windows-1252'], true);
        if ($encoding && $encoding !== 'UTF-8') {
            $string = mb_convert_encoding($string, 'UTF-8', $encoding);
        }

        // Remove any invalid UTF-8 sequences
        $string = mb_convert_encoding($string, 'UTF-8', 'UTF-8');

        // Replace control characters and other problematic characters
        $string = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $string);

        // Replace common HTML entities
        $string = html_entity_decode($string, ENT_QUOTES | ENT_HTML401, 'UTF-8');

        // Remove any remaining non-printable characters
        $string = preg_replace('/[^\P{C}\s]/u', '', $string);

        // Final validation and fallback
        if (!mb_check_encoding($string, 'UTF-8')) {
            // If still invalid, use only ASCII characters
            $string = preg_replace('/[^\x20-\x7E]/', '?', $string);
        }

        return trim($string);
    }

    /**
     * Clean an array of strings for JSON encoding
     */
    protected function cleanErrorDetailsArray($errorDetails)
    {
        if (!is_array($errorDetails)) {
            return [];
        }

        return array_map(function($detail) {
            return $this->cleanUtf8($detail);
        }, $errorDetails);
    }
}