<?php

namespace App\Models\Warehouse;

use App\Models\Department;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class LpoManagement extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'lpo_management';

    protected $fillable = [
        'lpo_number',
        'supplier_id',
        'project_id',
        'division_id',
        'lpo_date',
        'expected_delivery_date',
        'status',
        'priority',

        // Financial Information
        'total_amount',
        'currency',
        'exchange_rate',
        'total_amount_aed',

        // Budget Validation
        'budget_allocated',
        'budget_consumed',
        'budget_approved',
        'budget_approved_by',

        // Approval Workflow
        'created_by',
        'approved_by',
        'approved_at',
        'approval_notes',

        // Delivery Tracking
        'total_items',
        'delivered_items',
        'delivery_percentage',

        'terms_and_conditions',
        'notes'
    ];

    protected $casts = [
        'lpo_date' => 'date',
        'expected_delivery_date' => 'date',
        'approved_at' => 'datetime',
        'total_amount' => 'decimal:2',
        'total_amount_aed' => 'decimal:2',
        'exchange_rate' => 'decimal:4',
        'budget_allocated' => 'decimal:2',
        'budget_consumed' => 'decimal:2',
        'delivery_percentage' => 'decimal:2',
        'budget_approved' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($lpo) {
            if (empty($lpo->lpo_number)) {
                $lpo->lpo_number = static::generateLpoNumber();
            }
        });

        static::saving(function ($lpo) {
            // Auto-calculate AED amount
            if ($lpo->currency !== 'AED') {
                $lpo->total_amount_aed = $lpo->total_amount * $lpo->exchange_rate;
            } else {
                $lpo->total_amount_aed = $lpo->total_amount;
            }

            // Update delivery percentage
            if ($lpo->total_items > 0) {
                $lpo->delivery_percentage = ($lpo->delivered_items / $lpo->total_items) * 100;
            }
        });
    }

    /**
     * Generate LPO number
     */
    public static function generateLpoNumber()
    {
        $year = now()->year;
        $lastLpo = static::whereYear('created_at', $year)->latest()->first();
        $sequence = $lastLpo ? (int) substr($lastLpo->lpo_number, -6) + 1 : 1;

        return 'LPO-' . $year . '-' . str_pad($sequence, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Get the supplier
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * Get the project
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    /**
     * Get the division
     */
    public function division(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'division_id');
    }

    /**
     * Get the creator
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the approver
     */
    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Get the budget approver
     */
    public function budgetApprover(): BelongsTo
    {
        return $this->belongsTo(User::class, 'budget_approved_by');
    }

    /**
     * Get all items in this LPO
     */
    public function items(): HasMany
    {
        return $this->hasMany(LpoItem::class, 'lpo_id');
    }

    /**
     * Get related incoming operations
     */
    public function incomingOperations(): HasMany
    {
        return $this->hasMany(IncomingOperation::class, 'lpo_number', 'lpo_number');
    }

    /**
     * Scope for status
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for priority
     */
    public function scopeWithPriority($query, $priority)
    {
        return $query->where('priority', $priority);
    }

    /**
     * Scope for project
     */
    public function scopeForProject($query, $projectId)
    {
        return $query->where('project_id', $projectId);
    }

    /**
     * Scope for supplier
     */
    public function scopeForSupplier($query, $supplierId)
    {
        return $query->where('supplier_id', $supplierId);
    }

    /**
     * Scope for overdue LPOs
     */
    public function scopeOverdue($query)
    {
        return $query->where('expected_delivery_date', '<', now())
                    ->whereNotIn('status', ['completed', 'cancelled']);
    }

    /**
     * Check if LPO is approved
     */
    public function isApproved()
    {
        return in_array($this->status, ['approved', 'sent_to_supplier', 'partially_delivered', 'completed']);
    }

    /**
     * Check if LPO is overdue
     */
    public function isOverdue()
    {
        return $this->expected_delivery_date < now() && !in_array($this->status, ['completed', 'cancelled']);
    }

    /**
     * Check if LPO is fully delivered
     */
    public function isFullyDelivered()
    {
        return $this->delivery_percentage >= 100;
    }

    /**
     * Check if budget is available
     */
    public function hasBudgetAvailable($amount = null)
    {
        if (!$this->budget_allocated) {
            return true; // No budget constraint
        }

        $requiredAmount = $amount ?: $this->total_amount_aed;
        return ($this->budget_consumed + $requiredAmount) <= $this->budget_allocated;
    }

    /**
     * Get remaining budget
     */
    public function getRemainingBudgetAttribute()
    {
        if (!$this->budget_allocated) {
            return null;
        }

        return $this->budget_allocated - $this->budget_consumed;
    }

    /**
     * Get budget utilization percentage
     */
    public function getBudgetUtilizationAttribute()
    {
        if (!$this->budget_allocated || $this->budget_allocated == 0) {
            return 0;
        }

        return ($this->budget_consumed / $this->budget_allocated) * 100;
    }

    /**
     * Get status badge class
     */
    public function getStatusBadgeClassAttribute()
    {
        return match($this->status) {
            'draft' => 'badge-secondary',
            'pending_approval' => 'badge-warning',
            'approved' => 'badge-info',
            'sent_to_supplier' => 'badge-primary',
            'partially_delivered' => 'badge-warning',
            'completed' => 'badge-success',
            'cancelled' => 'badge-danger',
            default => 'badge-secondary'
        };
    }

    /**
     * Get priority badge class
     */
    public function getPriorityBadgeClassAttribute()
    {
        return match($this->priority) {
            'low' => 'badge-light',
            'medium' => 'badge-info',
            'high' => 'badge-warning',
            'urgent' => 'badge-danger',
            default => 'badge-light'
        };
    }

    /**
     * Get formatted total amount
     */
    public function getFormattedTotalAmountAttribute()
    {
        return $this->currency . ' ' . number_format($this->total_amount, 2);
    }

    /**
     * Get formatted total amount in AED
     */
    public function getFormattedTotalAmountAedAttribute()
    {
        return 'AED ' . number_format($this->total_amount_aed, 2);
    }

    /**
     * Get formatted remaining budget
     */
    public function getFormattedRemainingBudgetAttribute()
    {
        if (!$this->budget_allocated) {
            return 'No Limit';
        }

        return 'AED ' . number_format($this->remaining_budget, 2);
    }

    /**
     * Update delivery status based on incoming operations
     */
    public function updateDeliveryStatus()
    {
        $this->delivered_items = $this->items()->sum('quantity_delivered');
        $this->total_items = $this->items()->sum('quantity_ordered');

        if ($this->total_items > 0) {
            $this->delivery_percentage = ($this->delivered_items / $this->total_items) * 100;

            if ($this->delivery_percentage >= 100) {
                $this->status = 'completed';
            } elseif ($this->delivery_percentage > 0) {
                $this->status = 'partially_delivered';
            }
        }

        $this->save();
    }

    /**
     * Approve the LPO
     */
    public function approve(User $approver, $notes = null)
    {
        if (!$this->budget_approved && $this->budget_allocated) {
            throw new \Exception('Budget must be approved before LPO approval');
        }

        $this->status = 'approved';
        $this->approved_by = $approver->id;
        $this->approved_at = now();
        $this->approval_notes = $notes;
        $this->save();
    }

    /**
     * Approve budget
     */
    public function approveBudget(User $approver)
    {
        if (!$this->hasBudgetAvailable()) {
            throw new \Exception('Insufficient budget available');
        }

        $this->budget_approved = true;
        $this->budget_approved_by = $approver->id;
        $this->save();
    }

    /**
     * Send to supplier
     */
    public function sendToSupplier()
    {
        if (!$this->isApproved()) {
            throw new \Exception('LPO must be approved before sending to supplier');
        }

        $this->status = 'sent_to_supplier';
        $this->save();
    }

    /**
     * Cancel the LPO
     */
    public function cancel($reason = null)
    {
        $this->status = 'cancelled';
        $this->notes = $this->notes . "\n\nCancellation Reason: " . $reason;
        $this->save();
    }
}